<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Yotpo compatibility by Amasty for Magento 2 (System)
 */

namespace Amasty\Yotpo\Model\Yotpo;

use Amasty\Yotpo\Api\Review\ManagementInterface;
use Amasty\Yotpo\Model\ConfigProvider;
use Amasty\Yotpo\Model\ResourceModel\YotpoReview as YotpoReviewResource;
use Amasty\Yotpo\Model\YotpoReview;
use Amasty\Yotpo\Model\ResourceModel\Config as ConfigData;
use Amasty\Yotpo\Model\ResourceModel\YotpoReview\Collection as YotpoReviewCollection;
use Amasty\Yotpo\Model\ResourceModel\YotpoReview\CollectionFactory as YotpoReviewCollectionFactory;
use Magento\Framework\Exception\LocalizedException;

class Management implements ManagementInterface
{
    /**
     * @var Client
     */
    private $client;

    /**
     * @var ConfigData
     */
    private $configData;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    /**
     * @var YotpoReviewCollectionFactory
     */
    private $yotpoReviewCollectionFactory;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var YotpoReviewResource
     */
    private $yotpoReviewResource;

    public function __construct(
        Client $client,
        ConfigData $configData,
        YotpoReviewCollectionFactory $yotpoReviewCollectionFactory,
        ConfigProvider $configProvider,
        YotpoReviewResource $yotpoReviewResource,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->client = $client;
        $this->configData = $configData;
        $this->logger = $logger;
        $this->yotpoReviewCollectionFactory = $yotpoReviewCollectionFactory;
        $this->configProvider = $configProvider;
        $this->yotpoReviewResource = $yotpoReviewResource;
    }

    /**
     * @inheritDoc
     */
    public function updateByWebhook(string $event, $data)
    {
        try {
            if (isset($data['app_key']) && isset($data['id'])) {
                $storeIds = $this->configData->readStoreIdByValue($data['app_key']);
                switch ($event) {
                    case Client::REVIEW_UPDATED:
                        $storeId = reset($storeIds);
                        $reviewData = $this->client->getReviewById($data['id'], $storeId);
                        $isDeleted = (bool) $data['deleted'] ?? false;
                        $this->updateReviewIndex($reviewData, $isDeleted, $storeIds);
                        break;
                }
            }
        } catch (LocalizedException $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * @param array $reviewData
     * @param bool $isDeleted
     * @param array $storeIds
     * @throws LocalizedException
     */
    private function updateReviewIndex($reviewData, $isDeleted, $storeIds)
    {
        if (isset($reviewData['external_product_id'])
            && isset($reviewData['score'])
            && isset($reviewData['published'])
            && isset($reviewData['created_at'])
        ) {
            $dataToUpdate = [];
            $createdAt = strtotime($reviewData['created_at']);
            /** @var YotpoReviewCollection $yotpoReviewCollection */
            $yotpoReviewCollection = $this->yotpoReviewCollectionFactory->create()
                ->addIdFilter([$reviewData['external_product_id']])
                ->addStoreFilter($storeIds);
            /** @var YotpoReview $yotpoReview */
            foreach ($yotpoReviewCollection as $key => $yotpoReview) {
                $lastTime = $this->configProvider->getYotpoLastTime($yotpoReview->getStoreId());
                if ($lastTime && $lastTime > $createdAt) {
                    $yotpoReview->calculateAverageScore([
                        'score' => $reviewData['score'],
                        'count' => 1
                    ], !$isDeleted);

                    $dataToUpdate[] = [
                        YotpoReviewResource::PRODUCT_ID => $yotpoReview->getProductId(),
                        YotpoReviewResource::RATING_SUMMARY => $yotpoReview->getRatingSummary(),
                        YotpoReviewResource::TOTAL_REVIEWS => $yotpoReview->getTotalReviews(),
                        YotpoReviewResource::STORE_ID => $yotpoReview->getStoreId()
                    ];
                }
            }
            if ($dataToUpdate) {
                $this->yotpoReviewResource->insertOnDuplicate($dataToUpdate);
            }
        }
    }
}
