<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Yotpo compatibility by Amasty for Magento 2 (System)
 */

namespace Amasty\Yotpo\Model\Yotpo;

use Amasty\Yotpo\Model\ConfigProvider;
use Laminas\Http\Request;
use Magento\Framework\HTTP\Adapter\Curl;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Module\Manager as ModuleManager;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\Url;
use Magento\Framework\Url\QueryParamsResolverInterface;
use Magento\Framework\Serialize\Serializer\Json as Serializer;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;

class Client
{
    public const AUTHORIZE_URL_PATH = 'socialconnect/yotpo/authorize';

    public const YOTPO_SECURED_API_URL  = 'https://developers.yotpo.com';
    public const ALL_PRODUCTS_PATH = 'v2/YOUR_APP_KEY/products';
    public const REVIEWS_PATH = 'v2/YOUR_APP_KEY/reviews';
    public const REVIEW_PATH = 'v2/YOUR_APP_KEY/reviews/ID_PARAM';
    public const WEBHOOK_PATH = 'v2/YOUR_APP_KEY/webhooks';
    public const PRODUCT_REVIEWS_PATH = 'v2/YOUR_APP_KEY/products/ID_PARAM/reviews';
    public const APP_KEY = 'YOUR_APP_KEY';
    public const ID_PARAM = 'ID_PARAM';
    public const PRODUCTS_COUNT = 100;
    public const REVIEWS_COUNT = 30;

    // Webhooks events
    public const REVIEW_UPDATED = 'review_updated';

    /**
     * @var Url
     */
    private $urlBuilder;

    /**
     * @var Curl
     */
    private $curl;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var QueryParamsResolverInterface
     */
    private $queryParamsResolver;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    /**
     * @var \Yotpo\Yotpo\Model\Config
     */
    private $yotpoConfig = null;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var array
     */
    private $accessTokens = [];

    /**
     * @var TimezoneInterface
     */
    private $timezone;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        Serializer $serializer,
        Curl $curl,
        Url $urlBuilder,
        QueryParamsResolverInterface $queryParamsResolver,
        ObjectManagerInterface $objectManager,
        StoreManagerInterface $storeManager,
        TimezoneInterface $timezone,
        ModuleManager $moduleManager,
        ConfigProvider $configProvider,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->urlBuilder = $urlBuilder;
        $this->curl = $curl;
        $this->serializer = $serializer;
        $this->queryParamsResolver = $queryParamsResolver;
        $this->logger = $logger;
        $this->objectManager = $objectManager;
        $this->storeManager = $storeManager;
        $this->timezone = $timezone;
        if ($moduleManager->isEnabled('Yotpo_Yotpo')) {
            $this->yotpoConfig = $objectManager->get(\Yotpo\Yotpo\Model\Config::class);
        }
        $this->configProvider = $configProvider;
    }

    /**
     * @param string $type
     * @return string|null
     */
    public function getWebhookUrl($type)
    {
        $url = null;
        switch ($type) {
            case self::REVIEW_UPDATED:
                $url = $this->urlBuilder->getUrl('rest/all/V1', [
                    '_type' => UrlInterface::URL_TYPE_WEB
                ]) . 'amyotpo/review/update/';
                break;
        }

        return $url;
    }

    /**
     * @param string $type
     * @param int $storeId
     */
    public function createWebhook($type, $storeId)
    {
        $postData = [
            'url' => $this->getWebhookUrl($type),
            'event_name' => $type
        ];

        $this->curl($this->updatePath(self::WEBHOOK_PATH, $storeId), Request::METHOD_POST, $postData, true);
    }

    /**
     * @param $storeId
     * @return string|null
     */
    public function getAccessTokenFromConfig($storeId)
    {
        if ($this->yotpoConfig && !isset($this->accessTokens[$storeId])) {
            $this->accessTokens[$storeId] = $this->configProvider->getAccessToken($storeId);
        }

        return isset($this->accessTokens[$storeId]) ? $this->accessTokens[$storeId] : null;
    }

    /**
     * @param int $storeId
     * @return array
     */
    public function collectReviews($storeId)
    {
        $reviews = [];
        $firstLoad = true;
        if ($this->yotpoConfig) {
            $lastTime = $this->configProvider->getYotpoLastTime($storeId);
            $this->createWebhook(self::REVIEW_UPDATED, $storeId);
            if ($lastTime) {
                $firstLoad = false;
                $this->fetchNewReviews($reviews, $lastTime, $storeId);
                $this->updateLastTime($storeId, $lastTime);
            } else {
                $this->getProductsInfo($reviews, $storeId);
                $this->updateLastTime($storeId);
            }
        }

        return [$firstLoad, $reviews];
    }

    /**
     * @param string $path
     * @return mixed
     */
    private function sendApiRequest($path)
    {
        $result = $this->curl($path, Request::METHOD_GET);

        return $this->unserialize($result);
    }

    /**
     * @param string $path
     * @param $storeId
     * @param array $getParams
     * @return string
     */
    private function updatePath($path, $storeId, $getParams = [])
    {
        $path = str_replace(
            self::APP_KEY,
            (string)$this->yotpoConfig->getAppKey($storeId, 'stores'),
            $path
        );
        if (isset($getParams['id'])) {
            $path = str_replace(
                self::ID_PARAM,
                $getParams['id'],
                $path
            );
        }
        $getParams['access_token'] = $this->getAccessTokenFromConfig($storeId);
        $path .= '?';
        $path .= http_build_query($getParams);

        return self::YOTPO_SECURED_API_URL . '/' . $path;
    }

    /**
     * @param array $reviews
     * @param int $storeId
     */
    public function getProductsInfo(&$reviews, $storeId)
    {
        $page = 1;
        while (true) {
            $productsInfo = $this->sendApiRequest($this->updatePath(
                self::ALL_PRODUCTS_PATH,
                $storeId,
                ['count' => self::PRODUCTS_COUNT, 'page' => $page]
            ));
            if (isset($productsInfo['products']['products']) && !empty($productsInfo['products']['products'])) {
                foreach ($productsInfo['products']['products'] as $product) {
                    $reviews[] = [
                        'product_id' => $product['external_product_id'],
                        'rating_summary' => $product['average_score'],
                        'store_id' => $storeId,
                        'total_reviews' => $product['total_reviews']
                    ];
                }
                $page++;
            } else {
                break;
            }
        }
    }

    /**
     * @param array $reviews
     * @param string $since
     * @param int $storeId
     */
    public function fetchNewReviews(&$reviews, &$since, $storeId)
    {
        $page = 1;
        $lastCreatedAt = null;
        while (true) {
            $productReviews = $this->sendApiRequest($this->updatePath(
                self::REVIEWS_PATH,
                $storeId,
                ['count' => self::REVIEWS_COUNT, 'page' => $page, 'sort' => 'time', 'direction' => 'desc']
            ));
            if (isset($productReviews['reviews']['reviews']) && !empty($productReviews['reviews']['reviews'])) {
                foreach ($productReviews['reviews']['reviews'] as $review) {
                    if (!isset($review['external_product_id']) || !isset($review['score'])) {
                        continue;
                    }
                    if (strtotime($review['created_at']) <= $since) {
                        break 2;
                    }
                    if (isset($reviews[$review['external_product_id']])) {
                        $reviews[$review['external_product_id']]['score'] += $review['score'];
                        $reviews[$review['external_product_id']]['count']++;
                    } else {
                        $reviews[$review['external_product_id']] = [
                            'score' => $review['score'],
                            'count' => 1,
                            'store_id' => $storeId
                        ];
                    }
                    if (!$lastCreatedAt) {
                        $lastCreatedAt = $review['created_at'];
                    }
                }
                $since = $review['created_at'];
                $page++;
            } else {
                break;
            }
        }
        if ($lastCreatedAt) {
            $since = strtotime($lastCreatedAt);
        }
    }

    /**
     * @param $reviewId
     * @param $storeId
     * @return array|null
     */
    public function getReviewById($reviewId, $storeId)
    {
        if ($this->isYotpoInstalled()) {
            $data = $this->sendApiRequest($this->updatePath(
                self::REVIEW_PATH,
                $storeId,
                ['id' => $reviewId]
            ));
        }

        return $data['review']['review'] ?? null;
    }

    /**
     * @param int $storeId
     * @param null|string $lastTime
     */
    private function updateLastTime($storeId, $lastTime = null)
    {
        if (!$lastTime) {
            $lastTime = $this->timezone->date()->format('U');
        }
        $this->configProvider->saveYotpoLastTime($lastTime, $storeId);
    }

    /**
     * @param string $url
     * @param string $method
     * @param array $bodyParams
     * @param bool $jsonData
     * @return string
     */
    private function curl($url, $method, $bodyParams = [], $jsonData = false)
    {
        $this->curl->setConfig(['header' => false]);

        if ($jsonData) {
            $headers = ['Content-Type:application/json'];
            $bodyParams = $this->serializer->serialize($bodyParams);
        } else {
            $headers = [];
            $bodyParams = implode('&', $bodyParams);
        }

        $this->curl->write(
            $method,
            $url,
            '1.1',
            $headers,
            $bodyParams
        );
        $result = $this->curl->read();
        $this->curl->close();

        return $result;
    }

    /**
     * @param $string
     * @return array|bool|float|int|mixed|string|null
     */
    private function unserialize($string)
    {
        try {
            return $this->serializer->unserialize($string);
        } catch (\Exception $exception) {
            $this->logger->error(__('Error with getting content from Yotpo API: %1', $string));
            return [];
        }
    }

    /**
     * @return bool
     */
    public function isYotpoInstalled()
    {
        return $this->yotpoConfig !== null;
    }

    /**
     * @param $storeId
     * @return string
     */
    public function getAuthorizeUrl($storeId)
    {
        return $this->configProvider->getAuthorizeHost()
            . self::AUTHORIZE_URL_PATH
            . '?'
            . http_build_query(['key' => $this->generateAuthorizationKey($storeId)]);
    }

    /**
     * @param $storeId
     * @return string
     */
    private function generateAuthorizationKey($storeId)
    {
        $data = [
            'referer' => $this->getRefererUrl(),
            'store_id' => $storeId,
            'internal_token' => $this->configProvider->getInternalToken(true)
        ];
        return base64_encode($this->serializer->serialize($data));
    }

    /**
     * @return string
     */
    private function getRefererUrl()
    {
        $storeId = $this->storeManager->getDefaultStoreView()->getId();
        return $this->storeManager->getStore($storeId)->getBaseUrl(UrlInterface::URL_TYPE_LINK, true);
    }

    /**
     * @return bool
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getIsRefererSecure()
    {
        return strpos($this->getRefererUrl(), 'https') === 0;
    }

    /**
     * include_nested - needed for load user info
     *
     * @param int $productId
     * @param int $storeId
     * @param int $page
     * @param int $pageSize
     * @return array
     */
    public function getProductReviews(int $productId, int $storeId, int $page, int $pageSize): array
    {
        $response = $this->sendApiRequest($this->updatePath(
            self::PRODUCT_REVIEWS_PATH,
            $storeId,
            ['count' => $pageSize, 'page' => $page, 'id' => $productId, 'include_nested' => true]
        ));

        return $response['reviews']['reviews'] ?? [];
    }
}
