<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Yotpo compatibility by Amasty for Magento 2 (System)
 */

namespace Amasty\Yotpo\Model\Indexer;

use Amasty\Yotpo\Model\Yotpo\Client as YotpoClient;
use Amasty\Yotpo\Model\YotpoReview as YotpoReviewModel;
use Amasty\Yotpo\Model\ResourceModel\YotpoReview as YotpoReviewResource;
use Amasty\Yotpo\Model\ResourceModel\YotpoReview\Collection as YotpoReviewCollection;
use Amasty\Yotpo\Model\ResourceModel\YotpoReview\CollectionFactory as YotpoReviewCollectionFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Indexer\ActionInterface as IndexerActionInterface;
use Magento\Framework\Mview\ActionInterface as MviewActionInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class YotpoReview implements IndexerActionInterface, MviewActionInterface
{
    /**
     * @var YotpoReviewCollectionFactory
     */
    private $yotpoReviewCollectionFactory;

    /**
     * @var YotpoReviewResource
     */
    private $yotpoReviewResource;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var YotpoClient
     */
    private $yotpoClient;

    public function __construct(
        YotpoClient $yotpoClient,
        YotpoReviewResource $yotpoReviewResource,
        YotpoReviewCollectionFactory $yotpoReviewCollectionFactory,
        StoreManagerInterface $storeManager,
        LoggerInterface $logger
    ) {
        $this->yotpoReviewCollectionFactory = $yotpoReviewCollectionFactory;
        $this->yotpoReviewResource = $yotpoReviewResource;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
        $this->yotpoClient = $yotpoClient;
    }

    /**
     * @inheritDoc
     */
    public function executeFull()
    {
        $this->doReindex();
    }

    /**
     * @inheritDoc
     */
    public function executeList(array $ids)
    {
        $this->executeFull();
    }

    /**
     * @inheritDoc
     */
    public function executeRow($id)
    {
        $this->executeFull();
    }

    /**
     * @inheritDoc
     */
    public function execute($ids)
    {
        $this->executeFull();
    }

    private function doReindex()
    {
        if ($this->yotpoClient->isYotpoInstalled()) {
            foreach ($this->storeManager->getStores() as $store) {
                $this->reindexStoreInfo($store->getId());
            }
        }
    }

    /**
     * @param int $storeId
     */
    private function reindexStoreInfo($storeId)
    {
        list($firstLoad, $insertData) = $this->yotpoClient->collectReviews($storeId);
        if ($firstLoad) {
            $this->insertData($insertData);
        } else {
            $insertData = $this->updateOldReviews($insertData, $storeId);
            $this->updateNewReviews($insertData);
        }
    }

    /**
     * @param array $insertData
     */
    private function insertData($insertData)
    {
        if (!empty($insertData)) {
            try {
                $this->yotpoReviewResource->insertOnDuplicate($insertData);
            } catch (LocalizedException $e) {
                $this->logger->error($e->getMessage());
            }
        }
    }

    /**
     * @param array $insertData
     */
    private function updateNewReviews($insertData)
    {
        if (!empty($insertData)) {
            $yoptoInfo = [];
            foreach ($insertData as $productId => $insertDatum) {
                $yoptoInfo[] = [
                    YotpoReviewResource::PRODUCT_ID => $productId,
                    YotpoReviewResource::RATING_SUMMARY => $this->getRatingSummary(
                        $insertDatum['score'],
                        $insertDatum['count']
                    ),
                    YotpoReviewResource::TOTAL_REVIEWS => $insertDatum['count'],
                    YotpoReviewResource::STORE_ID => $insertDatum['store_id']
                ];
            }
            $this->insertData($yoptoInfo);
        }
    }

    /**
     * Returning part with new reviews
     *
     * @param array $insertData
     * @param $storeId
     * @return array
     */
    private function updateOldReviews($insertData, $storeId)
    {
        $dataToUpdate = [];
        /** @var YotpoReviewCollection $topratedCollection */
        $yotpoReviewCollection = $this->yotpoReviewCollectionFactory
            ->create()
            ->addIdFilter(array_keys($insertData))
            ->addStoreFilter([$storeId]);
        /** @var YotpoReviewModel $yotpoReview */
        foreach ($yotpoReviewCollection as $yotpoReview) {
            $scoreInfo = $insertData[$yotpoReview->getProductId()];
            $yotpoReview->calculateAverageScore($scoreInfo);
            $dataToUpdate[] = [
                YotpoReviewResource::PRODUCT_ID => $yotpoReview->getProductId(),
                YotpoReviewResource::RATING_SUMMARY => $yotpoReview->getRatingSummary(),
                YotpoReviewResource::TOTAL_REVIEWS => $yotpoReview->getTotalReviews(),
                YotpoReviewResource::STORE_ID => $storeId
            ];
            unset($insertData[$yotpoReview->getProductId()]);
        }
        if ($dataToUpdate) {
            $this->insertData($dataToUpdate);
        }

        return $insertData;
    }

    /**
     * @param $score
     * @param $total
     * @return float
     */
    private function getRatingSummary($score, $total)
    {
        if ($total) {
            $result = round($score / $total, 2);
        } else {
            $result = 0;
        }

        return $result;
    }
}
