<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Yotpo compatibility by Amasty for Magento 2 (System)
 */

namespace Amasty\Yotpo\Model;

use Amasty\Base\Model\ConfigProviderAbstract;
use Magento\Framework\App\Config\ReinitableConfigInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\Store;

class ConfigProvider extends ConfigProviderAbstract
{
    /**
     * @var string
     */
    protected $pathPrefix = 'amasty_yotpo/';

    public const ACCESS_TOKEN = 'general/access_token';
    public const AUTHORIZE_HOST = 'general/authorize_host';
    public const YOTPO_LAST_TIME_UPDATED = 'general/last_time';
    public const INTERNAL_TOKEN = 'general/internal_token';

    /**
     * @var WriterInterface
     */
    private $configWriter;

    /**
     * @var ReinitableConfigInterface
     */
    private $reinitableConfig;

    /**
     * @var EncryptorInterface
     */
    private $encryptor;

    public function __construct(
        ReinitableConfigInterface $reinitableConfig,
        ScopeConfigInterface $scopeConfig,
        WriterInterface $configWriter,
        EncryptorInterface $encryptor
    ) {
        parent::__construct($scopeConfig);
        $this->configWriter = $configWriter;
        $this->reinitableConfig = $reinitableConfig;
        $this->encryptor = $encryptor;
    }

    /**
     * @inheritdoc
     */
    public function getAccessToken($storeId = Store::DEFAULT_STORE_ID)
    {
        if ($accessToken = $this->getValue(self::ACCESS_TOKEN, $storeId)) {
            $accessToken = $this->encryptor->decrypt($accessToken);
        }

        return $accessToken;
    }

    /**
     * @param string $accessToken
     * @param null|int $storeId
     */
    public function saveAccessToken($accessToken, $storeId = null)
    {
        $this->saveData(
            self::ACCESS_TOKEN,
            $this->encryptor->encrypt($accessToken),
            $storeId
        );
    }

    /**
     * @return WriterInterface
     */
    public function getConfigWriter()
    {
        return $this->configWriter;
    }

    /**
     * @return ReinitableConfigInterface
     */
    public function getReinitableConfig()
    {
        return $this->reinitableConfig;
    }

    /**
     * @return mixed
     */
    public function getAuthorizeHost()
    {
        return $this->getValue(self::AUTHORIZE_HOST);
    }

    /**
     * @param string $lastTime
     * @param null|int $storeId
     */
    public function saveYotpoLastTime($lastTime, $storeId = null)
    {
        $this->saveData(self::YOTPO_LAST_TIME_UPDATED, $lastTime, $storeId);
    }

    /**
     * @param null $storeId
     * @return string
     */
    public function getYotpoLastTime($storeId = null)
    {
        return $this->getValue(self::YOTPO_LAST_TIME_UPDATED, $storeId);
    }

    /**
     * @param string $key
     * @param string $value
     * @param null $storeId
     */
    public function saveData($key, $value, $storeId = null)
    {
        if ($storeId) {
            $this->getConfigWriter()->save(
                $this->pathPrefix . $key,
                $value,
                ScopeInterface::SCOPE_STORES,
                $storeId
            );
        } else {
            $this->getConfigWriter()->save($this->pathPrefix . $key, $value);
        }

        $this->getReinitableConfig()->reinit();
    }

    /**
     * @param bool $needGenerate = false
     * @return string
     */
    public function getInternalToken($needGenerate = false)
    {
        $token = $this->getValue(self::INTERNAL_TOKEN);
        if ($needGenerate) {
            $token = hash('sha256', rand(1, PHP_INT_MAX));
            $this->getConfigWriter()->save($this->pathPrefix . self::INTERNAL_TOKEN, $token);
            $this->getReinitableConfig()->reinit();
        }

        return $token;
    }
}
