<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Landing Pages GraphQl for Magento 2 (System)
 */

namespace Amasty\XlandingGraphQl\Model;

use Amasty\Base\Model\Serializer;
use Amasty\Xlanding\Api\Data\PageInterface;
use Amasty\Xlanding\Api\PageRepositoryInterface;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Cms\Api\Data\BlockInterface;
use Magento\CmsGraphQl\Model\Resolver\DataProvider\Block as BlockDataProvider;
use Magento\Framework\DataObject;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Widget\Model\Template\Filter;

class ExtractDataFromPage
{
    /**
     * @var PageRepositoryInterface
     */
    private $pageRepository;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var CategoryRepository
     */
    private $categoryRepository;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var DataObjectFactory
     */
    private $dataObjectFactory;

    /**
     * @var BlockDataProvider
     */
    private $blockDataProvider;

    /**
     * @var Filter
     */
    private $filter;

    public function __construct(
        PageRepositoryInterface $pageRepository,
        Serializer $serializer,
        CategoryRepository $categoryRepository,
        StoreManagerInterface $storeManager,
        DataObjectFactory $dataObjectFactory,
        BlockDataProvider $blockDataProvider,
        Filter $filter
    ) {
        $this->pageRepository = $pageRepository;
        $this->serializer = $serializer;
        $this->categoryRepository = $categoryRepository;
        $this->storeManager = $storeManager;
        $this->dataObjectFactory = $dataObjectFactory;
        $this->blockDataProvider = $blockDataProvider;
        $this->filter = $filter;
    }

    public function extract(int $pageId, int $storeId): DataObject
    {
        $page = $this->pageRepository->getById($pageId);

        if (!$this->validateStore($page, $storeId)) {
            throw new GraphQlInputException(__('The requested Landing Page doesn\'t exist.'));
        }

        return $this->prepareData($page);
    }

    private function prepareData(PageInterface $page): DataObject
    {
        $pageData = $this->dataObjectFactory->create()->setData($page->getData());
        $pageData->setData('layout_file', $page->getLayoutFileUrl(false));
        $this->prepareStaticBlocks($page, $pageData);
        $this->prepareDescription($page, $pageData);
        $this->prepareMeta($page, $pageData);
        $this->prepareCategoryUrl($page, $pageData);
        $pageData->setData('type_id', XlandingTypeResolver::AMASTY_XLANDING);

        return $pageData;
    }

    private function prepareDescription(PageInterface $page, DataObject &$pageData)
    {
        if ($page->getLayoutTopDescription()) {
            $pageData->setData('layout_top_description', $this->parseWysiwyg($page->getLayoutTopDescription()));
        }
        if ($page->getLayoutBottomDescription()) {
            $pageData->setData('layout_bottom_description', $this->parseWysiwyg($page->getLayoutBottomDescription()));
        }
    }

    private function parseWysiwyg(string $content): string
    {
        $content = $this->filter->filter($content);
        $store = $this->storeManager->getStore();
        $baseMediaUrl = $store->getBaseUrl(UrlInterface::URL_TYPE_MEDIA);

        return str_replace($baseMediaUrl, '/' . UrlInterface::URL_TYPE_MEDIA . '/', $content);
    }

    private function prepareCategoryUrl(PageInterface $page, DataObject &$pageData)
    {
        $categoryId = $page->getDynamicCategoryId();
        if ($categoryId) {
            $category = $this->categoryRepository->get($categoryId);
            if ($category) {
                $baseUrl = $this->storeManager->getStore()->getBaseUrl();
                $categoryUrl = $category->getUrl();
                $pageData->setData('dynamic_category_url', str_replace($baseUrl, "", $categoryUrl));
            }
        }
    }

    private function prepareStaticBlocks(PageInterface $page, DataObject &$pageData)
    {
        $top = $page->getLayoutStaticTop();
        if ($top) {
            $pageData->setData('layout_static_top', $this->getCmsBlockHtml($top));
        }

        $bottom = $page->getLayoutStaticBottom();
        if ($bottom) {
            $pageData->setData('layout_static_bottom', $this->getCmsBlockHtml($bottom));
        }
    }

    private function getCmsBlockHtml(string $blockId): string
    {
        try {
            $data = $this->blockDataProvider->getBlockById(
                (int)$blockId,
                (int)$this->storeManager->getStore()->getId()
            );
        } catch (NoSuchEntityException $exception) {
            return '';
        }

        return $data[BlockInterface::CONTENT];
    }

    private function prepareMeta(PageInterface $page, DataObject &$pageData)
    {
        $meta = [];
        foreach ($page->getMetaData() as $storeData) {
            foreach ($storeData as $key => $metaData) {
                $storeData[substr($key, 0, -2)] = $metaData;
                unset($storeData[$key]);
            }
            $meta[] = $storeData;
        }
        $pageData->setData('meta_data', $this->serializer->serialize($meta));
    }

    private function validateStore(PageInterface $page, int $storeId): bool
    {
        return in_array(Store::DEFAULT_STORE_ID, $page->getStoreId())
            || in_array($storeId, $page->getStoreId());
    }
}
