<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Landing Pages for Magento 2
 */

namespace Amasty\Xlanding\Setup\Patch\DeclarativeSchemaApplyBefore;

use Amasty\Base\Model\Serializer;
use Amasty\Xlanding\Api\Data\PageInterface;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Setup\Patch\SchemaPatchInterface;
use Magento\Framework\Setup\SchemaSetupInterface;
use Magento\Store\Model\StoreManagerInterface;

class ChangeMetaData implements SchemaPatchInterface
{
    /**
     * Information from these columns will be written to the meta_data column
     *
     * @var string[]
     */
    private $columnsToTransfer = [
        'meta_title',
        'meta_keywords',
        'meta_description'
    ];

    /**
     * @var SchemaSetupInterface
     */
    private $setup;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Serializer
     */
    private $serializer;

    public function __construct(
        SchemaSetupInterface $setup,
        StoreManagerInterface $storeManager,
        Serializer $serializer
    ) {
        $this->setup = $setup;
        $this->storeManager = $storeManager;
        $this->serializer = $serializer;
    }

    public function apply(): self
    {
        $connection = $this->setup->getConnection();
        $landingPageTable = $this->setup->getTable(PageInterface::MAIN_TABLE);

        if ($this->setup->tableExists($landingPageTable) && $this->canApply($connection, $landingPageTable)) {
            $this->addMetaDataColumn($connection, $landingPageTable);
            $columnsToSelect = $this->columnsToTransfer;
            $columnsToSelect[] = 'page_id';
            $select = $connection->select()->from($landingPageTable, $columnsToSelect);

            foreach ($connection->fetchAll($select) as $page) {
                $metaData = [];

                foreach ($this->storeManager->getStores() as $store) {
                    $metaData[$store->getId()] = [
                        'meta_title_' . $store->getId() => $page['meta_title'],
                        'meta_keywords_' . $store->getId() => $page['meta_keywords'],
                        'meta_description_' . $store->getId() => $page['meta_description']
                    ];
                }

                $metaData = $this->serializer->serialize($metaData);
                $sql = 'UPDATE ' . $landingPageTable . ' SET `meta_data` = :metaData WHERE `page_id` = :pageId;';
                $connection->query($sql, [
                    'metaData' => $metaData,
                    'pageId' => $page['page_id']
                ]);
            }
        }

        return $this;
    }

    private function addMetaDataColumn(AdapterInterface $connection, string $table): void
    {
        if (!$connection->tableColumnExists($table, PageInterface::META_DATA)) {
            $connection->addColumn(
                $table,
                PageInterface::META_DATA,
                [
                    'type' => \Magento\Framework\DB\Ddl\Table::TYPE_TEXT,
                    'nullable' => true,
                    'comment' => 'Meta data'
                ]
            );

        }
    }

    private function canApply(AdapterInterface $connection, string $table): bool
    {
        $canApply = true;

        foreach ($this->columnsToTransfer as $column) {
            if (!$connection->tableColumnExists($table, $column)) {
                $canApply = false;
                break;
            }
        }

        return $canApply;
    }

    public function getAliases(): array
    {
        return [];
    }

    public static function getDependencies(): array
    {
        return [];
    }
}
