<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Landing Pages for Magento 2
 */

namespace Amasty\Xlanding\Model\XmlSitemap\DataProvider;

use Amasty\Xlanding\Model\ResourceModel\PageFactory;
use Amasty\XmlSitemap\Api\SitemapInterface;
use Magento\Framework\Phrase;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Amasty_XmlSitemap entity provider
 */
class Landing
{
    private const ENTITY_CODE = 'amasty_xlanding';

    /**
     * @var PageFactory
     */
    private $pageFactory;

    /**
     * @var string
     */
    private $baseUrl;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    public function __construct(
        PageFactory $pageFactory,
        StoreManagerInterface $storeManager
    ) {
        $this->pageFactory = $pageFactory;
        $this->storeManager = $storeManager;
    }

    public function getData(SitemapInterface $sitemap): \Generator
    {
        $sitemapEntityData = $sitemap->getEntityData($this->getEntityCode());

        foreach ($this->getLandingCollection($sitemap->getStoreId()) as $landingPage) {
            /** @var  $landingPage \Amasty\Xlanding\Model\Page */
            $pageUrl = $this->getBaseUrl($sitemap->getStoreId()) . $landingPage->getUrl();

            yield [
                [
                    'loc'       => $pageUrl,
                    'frequency' => $sitemapEntityData->getFrequency(),
                    'priority'  => $sitemapEntityData->getPriority()
                ]
            ];
        }
    }

    private function getLandingCollection(int $storeId): array
    {
        $page = $this->pageFactory->create();

        return $page->getSitemapCollection($storeId);
    }

    private function getBaseUrl(int $storeId): string
    {
        if (!$this->baseUrl) {
            $store = $this->storeManager->getStore($storeId);
            $isSecure = $store->isUrlSecure();
            $this->baseUrl = $store->getBaseUrl(UrlInterface::URL_TYPE_LINK, $isSecure);
            $this->baseUrl = rtrim($this->baseUrl, '/') . '/';
        }

        return $this->baseUrl;
    }

    public function getEntityCode(): string
    {
        return self::ENTITY_CODE;
    }

    public function getEntityLabel(): Phrase
    {
        return __('Amasty Landing Pages');
    }
}
