<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Custom Stock Status GraphQl for Magento 2 (System)
 */

namespace Amasty\StockstatusGraphQl\Model\Resolver;

use Amasty\Stockstatus\Model\Stockstatus\Processor;
use Amasty\StockstatusGraphQl\Model\Customer\Emulator as CustomerEmulator;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;

class DataProvider
{
    /**
     * @var Processor
     */
    private $processor;

    /**
     * @var CustomerEmulator
     */
    private $customerEmulator;

    public function __construct(
        Processor $processor,
        CustomerEmulator $customerEmulator
    ) {
        $this->processor = $processor;
        $this->customerEmulator = $customerEmulator;
    }

    /**
     * @param ProductInterface $product
     * @param ContextInterface $context
     * @return array
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function getStockData(ProductInterface $product, ContextInterface $context): array
    {
        $this->customerEmulator->start($context->getUserId());
        $this->processor->execute([$product]);
        $this->customerEmulator->stop();

        $stockStatusInformation = $product->getExtensionAttributes()->getStockstatusInformation();
        $baseUrl = $context->getExtensionAttributes()->getStore()->getBaseUrl();

        return [
            'message' => $stockStatusInformation->getStatusMessage(),
            'status_icon' => $this->getRelativePath((string) $stockStatusInformation->getStatusIcon(), $baseUrl),
            'tooltip' => $stockStatusInformation->getTooltipText()
        ];
    }

    private function getRelativePath(string $path, string $baseUrl)
    {
        return str_replace($baseUrl, '', $path);
    }
}
