<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Product Availability Status Base for Magento 2
 */

namespace Amasty\Stockstatus\Setup\Patch\Schema;

use Amasty\Stockstatus\Api\Data\IconInterface;
use Amasty\Stockstatus\Api\Data\StockstatusSettingsInterface;
use Amasty\Stockstatus\Model\Icon\GetMediaPath;
use Exception;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\DB\Select;
use Magento\Framework\Filesystem;
use Magento\Framework\Setup\Patch\SchemaPatchInterface;
use Magento\Framework\Setup\SchemaSetupInterface;
use Magento\Store\Model\Store;
use Psr\Log\LoggerInterface;

/**
 * Move data from old table to new settings table
 */
class MoveIconDataToSettings implements SchemaPatchInterface
{
    /**
     * @var SchemaSetupInterface
     */
    private $setup;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @var GetMediaPath
     */
    private $getMediaPath;

    public function __construct(
        SchemaSetupInterface $setup,
        LoggerInterface $logger,
        Filesystem $filesystem,
        GetMediaPath $getMediaPath
    ) {
        $this->setup = $setup;
        $this->logger = $logger;
        $this->filesystem = $filesystem;
        $this->getMediaPath = $getMediaPath;
    }

    public function apply(): self
    {
        $connection = $this->setup->getConnection();
        $iconTable = $this->setup->getTable(IconInterface::MAIN_TABLE);
        $stockStatusSettingsTable = $this->setup->getTable(StockstatusSettingsInterface::MAIN_TABLE);
        $settingsSelect = $connection->select()->from($stockStatusSettingsTable)->limit(1);

        if ($this->setup->tableExists($iconTable)) {
            try {
                $connection->beginTransaction();
                $dataToMove = array_map(
                    [$this, 'prepareData'],
                    $this->getIconsData($connection, $iconTable)
                );

                if (!empty($dataToMove)) {
                    $connection->insertMultiple(
                        $stockStatusSettingsTable,
                        $dataToMove
                    );
                }
            } catch (Exception $e) {
                $connection->rollBack();
                $this->logger->error($e->getMessage());
            }

            $connection->commit();
            $connection->dropTable($iconTable);
        } elseif (!$connection->fetchCol($settingsSelect)) {
            $this->saveOldImages();
        }

        return $this;
    }

    private function saveOldImages(): void
    {
        $imagesToInsert = [];
        $oldImages = $this->filesystem->getDirectoryReadByPath($this->getMediaPath->execute())->read();

        foreach ($oldImages as $oldImage) {
            preg_match('@(\d+)\.(jpg|jpeg|gif|png)@', $oldImage, $matches);

            if (isset($matches[1])) {
                $imagesToInsert[] = [
                    StockstatusSettingsInterface::OPTION_ID => $matches[1],
                    StockstatusSettingsInterface::STORE_ID => Store::DEFAULT_STORE_ID,
                    StockstatusSettingsInterface::IMAGE_PATH => $oldImage
                ];
            }
            if (count($imagesToInsert) > 100) {
                $this->insertData($imagesToInsert);
                $imagesToInsert = [];
            }
        }

        if (!empty($imagesToInsert)) {
            $this->insertData($imagesToInsert);
        }
    }

    /**
     * Map Icon Data to StockstatusSettings
     *
     * @param array $iconData
     * @return array
     */
    private function prepareData(array $iconData): array
    {
        return [
            StockstatusSettingsInterface::STORE_ID => (int)$iconData['store_id'],
            StockstatusSettingsInterface::OPTION_ID => (int)$iconData['option_id'],
            StockstatusSettingsInterface::IMAGE_PATH => $iconData['path']
        ];
    }

    private function insertData(array $data): void
    {
        $this->setup->getConnection()->insertArray(
            $this->setup->getTable(StockstatusSettingsInterface::MAIN_TABLE),
            [
                StockstatusSettingsInterface::OPTION_ID,
                StockstatusSettingsInterface::STORE_ID,
                StockstatusSettingsInterface::IMAGE_PATH
            ],
            $data,
            AdapterInterface::INSERT_IGNORE
        );
    }

    private function getIconsData(AdapterInterface $connection, string $tableName): array
    {
        $select = $connection->select();
        $select->from($tableName);
        $select->order('store_id ' . Select::SQL_ASC);

        return $connection->fetchAssoc($select);
    }

    public function getAliases(): array
    {
        return [];
    }

    public static function getDependencies(): array
    {
        return [];
    }
}
