<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Product Availability Status Base for Magento 2
 */

namespace Amasty\Stockstatus\Setup\Patch\Data;

use Amasty\Stockstatus\Model\Attribute\Creator;
use Amasty\Stockstatus\Model\Source\StockStatus;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\Eav\Api\AttributeOptionManagementInterface;
use Magento\Eav\Api\Data\AttributeOptionInterfaceFactory;
use Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory;
use Magento\Eav\Model\Entity\Attribute\Source\Table as SourceTable;
use Magento\Eav\Setup\EavSetup;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Framework\Setup\Patch\PatchRevertableInterface;
use Magento\Store\Model\Store;

class CreateStockStatusAttribute implements DataPatchInterface, PatchRevertableInterface
{
    private const STOCK_STATUS_NOTE = 'The current setting has the highest priority - a rule '
            . '(if applied to current product) will be ignored.';

    /**
     * @var array
     */
    private $examples = [
        'ONLY {qty} LEFT!',
        'On Sale {special_price}',
        'Will be available {day-after-tomorrow}',
        'Backorder',
        'Product will be available on {expected_date}'
    ];

    /**
     * @var AttributeOptionManagementInterface
     */
    private $attributeOptionManagement;

    /**
     * @var AttributeOptionInterfaceFactory
     */
    private $attributeOptionFactory;

    /**
     * @var AttributeOptionLabelInterfaceFactory
     */
    private $optionLabelFactory;

    /**
     * @var ProductAttributeRepositoryInterface
     */
    private $attributeRepository;

    /**
     * @var EavSetup
     */
    private $eavSetup;

    /**
     * @var Creator
     */
    private $attributeCreator;

    public function __construct(
        EavSetupFactory $eavSetupFactory,
        ModuleDataSetupInterface $setup,
        Creator $attributeCreator,
        ProductAttributeRepositoryInterface $attributeRepository,
        AttributeOptionManagementInterface $attributeOptionManagement,
        AttributeOptionLabelInterfaceFactory $optionLabelFactory,
        AttributeOptionInterfaceFactory $attributeOptionFactory
    ) {
        $this->eavSetup = $eavSetupFactory->create(['setup' => $setup]);
        $this->attributeCreator = $attributeCreator;
        $this->attributeOptionManagement = $attributeOptionManagement;
        $this->attributeOptionFactory = $attributeOptionFactory;
        $this->optionLabelFactory = $optionLabelFactory;
        $this->attributeRepository = $attributeRepository;
    }

    public function apply(): self
    {
        if (!$this->eavSetup->getAttribute(Product::ENTITY, StockStatus::ATTIRUBTE_CODE)) {
            $this->attributeCreator->createProductAttribute(
                StockStatus::ATTIRUBTE_CODE,
                'Custom Stock Status',
                [
                    'source' => SourceTable::class,
                    'note' => self::STOCK_STATUS_NOTE
                ]
            );

            $this->addExamples();
        } else {
            $attributeIdStockStatus = $this->eavSetup->getAttributeId(
                Product::ENTITY,
                StockStatus::ATTIRUBTE_CODE
            );
            $this->eavSetup->updateAttribute(
                Product::ENTITY,
                $attributeIdStockStatus,
                'note',
                self::STOCK_STATUS_NOTE
            );
        }

        return $this;
    }

    private function addExamples(): void
    {
        $attribute = $this->attributeRepository->get(StockStatus::ATTIRUBTE_CODE);
        $attributeId = $attribute->getAttributeId();

        foreach ($this->examples as $example) {
            $optionLabel = $this->optionLabelFactory->create();
            $optionLabel->setStoreId(Store::DEFAULT_STORE_ID);
            $optionLabel->setLabel($example);

            $option = $this->attributeOptionFactory->create();
            $option->setLabel($example);
            $option->setStoreLabels([$optionLabel]);
            $option->setSortOrder(0);
            $option->setIsDefault(false);

            $this->attributeOptionManagement->add(
                Product::ENTITY,
                $attributeId,
                $option
            );
        }
    }

    public function getAliases(): array
    {
        return [];
    }

    public static function getDependencies(): array
    {
        return [];
    }

    public function revert(): void
    {
        $this->eavSetup->removeAttribute(Product::ENTITY, StockStatus::ATTIRUBTE_CODE);
    }
}
