<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Product Availability Status Base for Magento 2
 */

namespace Amasty\Stockstatus\Model\Stockstatus\Utils;

use Amasty\Stockstatus\Model\Stockstatus\Formatter;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\ResourceModel\Product as ProductResource;
use Magento\Framework\Pricing\PriceCurrencyInterface;

class GetAttributeValue
{
    /**
     * @var FormatDate
     */
    private $formatDate;

    /**
     * @var PriceCurrencyInterface
     */
    private $priceCurrency;

    /**
     * @var ProductResource
     */
    private $productResource;

    public function __construct(
        FormatDate $formatDate,
        PriceCurrencyInterface $priceCurrency,
        ProductResource $productResource
    ) {
        $this->formatDate = $formatDate;
        $this->priceCurrency = $priceCurrency;
        $this->productResource = $productResource;
    }

    /**
     * @param Product $product
     * @param string $attributeCode
     * @return float|string
     */
    public function execute(Product $product, string $attributeCode)
    {
        $result = '';
        $value = ($product->hasData($attributeCode)
            ? $product->getData($attributeCode)
            : $this->productResource->getAttributeRawValue(
                $product->getId(),
                $attributeCode,
                $product->getStoreId()
            )
        );

        if ($value) {
            $attribute = $product->getResource()->getAttribute($attributeCode);
            if ($attribute && $attribute->usesSource()) {
                $result = $attribute->getSource()->getOptionText($value);
            } elseif (preg_match("/([0-9]{4})-([0-9]{2})-([0-9]{2})/", $value)) {
                $result = $this->formatDate->format($value, Formatter::DEFAULT_DATE_FORMAT);
            } elseif ($attribute->getFrontendInput() == 'price') {
                $result = $this->priceCurrency->format($value, false);
            } else {
                $result = $value;
            }
        }

        return is_array($result) ? implode(',', $result) : $result;
    }
}
