<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Shop by SEO Redirect
 */

namespace Amasty\ShopByRedirect\Model;

use Amasty\ShopbyBase\Model\UrlBuilder\UrlModifier;
use Amasty\ShopbySeo\Model\ConfigProvider as SeoConfigProvider;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\Result\RedirectFactory;
use Magento\Framework\UrlInterface;

class RedirectResolver
{
    /**
     * @var UrlModifier
     */
    private $urlModifier;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var SeoConfigProvider
     */
    private $seoConfigProvider;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var UrlInterface
     */
    private $urlService;

    /**
     * @var RedirectFactory
     */
    private $redirectFactory;

    /**
     * @var array
     */
    private $notAllowedParams;

    public function __construct(
        UrlModifier $urlModifier,
        ConfigProvider $configProvider,
        SeoConfigProvider $seoConfigProvider,
        RequestInterface $request,
        UrlInterface $urlService,
        RedirectFactory $redirectFactory,
        array $notAllowedParams = []
    ) {
        $this->urlModifier = $urlModifier;
        $this->configProvider = $configProvider;
        $this->seoConfigProvider = $seoConfigProvider;
        $this->request = $request;
        $this->urlService = $urlService;
        $this->redirectFactory = $redirectFactory;
        $this->notAllowedParams = $notAllowedParams;
    }

    public function isRedirectAllowed(): bool
    {
        foreach ($this->notAllowedParams as $notAllowedParam) {
            if ($this->request->getParam($notAllowedParam, false)) {
                return false;
            }
        }

        return $this->configProvider->isRedirectToSeo() && $this->seoConfigProvider->isSeoUrlEnabled();
    }

    public function resolveRedirect(int $categoryId): ?Redirect
    {
        $currentUrl = $this->urlService->getCurrentUrl();
        $redirectUrl = $this->urlModifier->execute($currentUrl, $categoryId);

        if ($currentUrl === $redirectUrl) {
            return null;
        }

        $redirect = $this->redirectFactory->create();
        $redirect->setUrl($redirectUrl);
        $redirect->setHttpResponseCode(301);

        return $redirect;
    }
}
