<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package RMA GraphQl for Magento 2 (System)
 */

namespace Amasty\RmaGraphQl\Model\Resolver\Order;

use Amasty\Rma\Api\Data\RequestInterface;
use Amasty\Rma\Api\Data\ReturnOrderInterface;
use Amasty\Rma\Api\Data\ReturnOrderItemInterface;
use Amasty\Rma\Model\OptionSource\NoReturnableReasons;
use Amasty\Rma\Model\Order\CreateReturnProcessor;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Sales\Api\Data\OrderItemInterface;

class GetReturnOrderItem implements ResolverInterface
{
    public const ALREADY_RETURNED_REQUESTS_KEY = 'already_returned_requests';

    /**
     * @var CreateReturnProcessor
     */
    private $createReturnProcessor;

    /**
     * @var array
     */
    private $returnOrdersCache = [];

    public function __construct(
        CreateReturnProcessor $createReturnProcessor
    ) {
        $this->createReturnProcessor = $createReturnProcessor;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!isset($value['model'])) {
            throw new GraphQlInputException(__('"model" value should be specified'));
        }
        /** @var OrderItemInterface $order */
        $orderItem = $value['model'];
        $orderId = $orderItem->getOrderId();

        if (!isset($this->returnOrdersCache[$orderId])) {
            $this->returnOrdersCache[$orderId] = $this->createReturnProcessor->process($orderId);
        }
        $returnOrder = $this->returnOrdersCache[$orderId];

        return $this->matchReturnItem($returnOrder, (int)$orderItem->getItemId());
    }

    private function matchReturnItem(ReturnOrderInterface $returnOrder, int $orderItemId): array
    {
        foreach ($returnOrder->getItems() as $returnOrderItem) {
            if ($returnOrderItem->getItem()->getId() == $orderItemId
                || $returnOrderItem->getItem()->getParentItemId() == $orderItemId
            ) {
                return $this->formatReturnItem($returnOrderItem);
            }
        }

        return [];
    }

    private function formatReturnItem(ReturnOrderItemInterface $returnOrderItem): array
    {
        $returnOrderItemData = $returnOrderItem->getData();
        $returnOrderItemData[ReturnOrderItemInterface::RESOLUTIONS] = array_map(function ($resolution) {
            return $resolution->getData();
        }, $returnOrderItem->getResolutions());

        if ($returnOrderItem->getNoReturnableReason() == NoReturnableReasons::ALREADY_RETURNED) {
            $returnOrderItemData[self::ALREADY_RETURNED_REQUESTS_KEY] = array_map(function ($request) {
                return $request[RequestInterface::REQUEST_ID];
            }, $returnOrderItem->getNoReturnableData());
        }
        $returnOrderItemData['order_item_id'] = $returnOrderItem->getItem()->getItemId();

        return $returnOrderItemData;
    }
}
