<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package RMA GraphQl for Magento 2 (System)
 */

namespace Amasty\RmaGraphQl\Model\Resolver\Customer;

use Amasty\Rma\Model\ReturnableOrdersProvider;
use Magento\CustomerGraphQl\Model\Customer\GetCustomer;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Sales\Api\Data\OrderInterface;

class ReturnableOrders implements ResolverInterface
{
    /**
     * @var GetCustomer
     */
    private $getCustomer;

    /**
     * @var ReturnableOrdersProvider
     */
    private $returnableOrderProvider;

    /**
     * @var TimezoneInterface
     */
    private $localeDate;

    /**
     * @var PriceCurrencyInterface
     */
    private $priceCurrency;

    public function __construct(
        GetCustomer $getCustomer,
        ReturnableOrdersProvider $returnableOrderProvider,
        PriceCurrencyInterface $priceCurrency,
        TimezoneInterface $localeDate
    ) {
        $this->getCustomer = $getCustomer;
        $this->returnableOrderProvider = $returnableOrderProvider;
        $this->localeDate = $localeDate;
        $this->priceCurrency = $priceCurrency;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $customer = $this->getCustomer->execute($context);
        $result = [];

        foreach ($this->returnableOrderProvider->getOrders((int)$customer->getId()) as $order) {
            $label = __(
                'Order #%1 - %2 - %3',
                $order[OrderInterface::INCREMENT_ID],
                $this->formatDate($order[OrderInterface::CREATED_AT]),
                $this->formatPrice((float)$order[OrderInterface::GRAND_TOTAL])
            )->render();

            $result[] = [
                'label' => $label,
                'order_id' => $order[OrderInterface::INCREMENT_ID]
                ];
        }

        return $result;
    }

    private function formatDate(string $date): string
    {
        $date = new \DateTime($date);

        return $this->localeDate->formatDateTime(
            $date,
            \IntlDateFormatter::SHORT,
            \IntlDateFormatter::NONE,
        );
    }

    private function formatPrice(float $price): string
    {
        return $this->priceCurrency->convertAndFormat(
            $price,
            false,
            2
        );
    }
}
