<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package RMA GraphQl for Magento 2 (System)
 */

namespace Amasty\RmaGraphQl\Model\Resolver\Customer;

use Amasty\Rma\Api\CustomerRequestRepositoryInterface;
use Amasty\Rma\Api\Data\RequestInterface;
use Amasty\Rma\Controller\FrontendRma;
use Amasty\RmaGraphQl\Model\Formatter\RequestData;
use Amasty\RmaGraphQl\Model\Request\CreateReturnRequest;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class PlaceReturnRequest implements ResolverInterface
{
    public const MESSAGE_KEY = 'comment';
    public const FILES_KEY = 'files';

    /**
     * @var FrontendRma
     */
    private $frontendRma;

    /**
     * @var CreateReturnRequest
     */
    private $createReturnRequest;

    /**
     * @var CustomerRequestRepositoryInterface
     */
    private $customerRequestRepository;

    /**
     * @var RequestData
     */
    private $requestDataFormatter;

    public function __construct(
        FrontendRma $frontendRma,
        CreateReturnRequest $createReturnRequest,
        CustomerRequestRepositoryInterface $customerRequestRepository,
        RequestData $requestDataFormatter
    ) {
        $this->frontendRma = $frontendRma;
        $this->createReturnRequest = $createReturnRequest;
        $this->customerRequestRepository = $customerRequestRepository;
        $this->requestDataFormatter = $requestDataFormatter;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$context->getExtensionAttributes()->getIsCustomer()) {
            throw new GraphQlAuthorizationException(__('Customer is not authorized.'));
        }
        $args['input']['return_items'] = array_filter($args['input']['return_items']);

        $returnRequest = $this->customerRequestRepository->create($this->createReturnRequest->execute($args));
        $this->processMessageData($returnRequest, $args['input']);

        return $this->requestDataFormatter->prepareRequestData(
            $returnRequest,
            (int)$context->getExtensionAttributes()->getStore()->getId()
        );
    }

    private function processMessageData(RequestInterface $returnRequest, array $inputData): void
    {
        $message = '';
        $files = [];

        if (isset($inputData[self::MESSAGE_KEY]) && !empty($inputData[self::MESSAGE_KEY])) {
            $message = $inputData[self::MESSAGE_KEY];
        }
        if (isset($inputData[self::FILES_KEY]) && !empty($inputData[self::FILES_KEY])) {
            $files = $inputData[self::FILES_KEY];
        }

        if (!empty($message) || !empty($files)) {
            $this->frontendRma->saveNewReturnMessage($returnRequest, $message, $files);
        }
    }
}
