<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package RMA GraphQl for Magento 2 (System)
 */

namespace Amasty\RmaGraphQl\Model\Request;

use Amasty\Rma\Api\CustomerRequestRepositoryInterface;
use Amasty\Rma\Api\Data\RequestInterface;
use Magento\CustomerGraphQl\Model\Customer\GetCustomer;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Exception\GraphQlAuthenticationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\Phrase;
use Magento\GraphQl\Model\Query\ContextInterface;

class ValidatedRequestProvider
{
    /**
     * @var GetCustomer
     */
    private $getCustomer;

    /**
     * @var CustomerRequestRepositoryInterface
     */
    private $requestRepository;

    public function __construct(
        GetCustomer $getCustomer,
        CustomerRequestRepositoryInterface $requestRepository
    ) {
        $this->getCustomer = $getCustomer;
        $this->requestRepository = $requestRepository;
    }

    /**
     * Retrieve return request by hash and validate is it allowed for current customer
     *
     * @param string $hash
     * @param ContextInterface $context
     * @param Phrase|null $errorMessage
     * @return RequestInterface
     * @throws GraphQlAuthenticationException
     * @throws GraphQlInputException
     */
    public function getByHash(string $hash, ContextInterface $context, Phrase $errorMessage = null): RequestInterface
    {
        try {
            $request = $this->requestRepository->getByHash($hash);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlInputException(__('Non existing hash was provided.'));
        }

        $customerId = $this->getCustomerId($context);
        if ($customerId != $request->getCustomerId()) {
            if ($errorMessage === null) {
                $errorMessage = __('Customer isn\'t authorized.');
            }
            throw new GraphQlAuthenticationException($errorMessage);
        }

        return $request;
    }

    /**
     * @param int $id
     * @param ContextInterface $context
     * @param Phrase|null $errorMessage
     * @return RequestInterface
     * @throws GraphQlAuthenticationException
     */
    public function getById(int $id, ContextInterface $context, Phrase $errorMessage = null): RequestInterface
    {
        $customerId = $this->getCustomerId($context);
        try {
            $request = $this->requestRepository->getById($id, (int)$customerId);
        } catch (NoSuchEntityException $e) {
            if ($errorMessage === null) {
                $errorMessage = __('Customer isn\'t authorized.');
            }
            throw new GraphQlAuthenticationException($errorMessage);
        }

        return $request;
    }

    private function getCustomerId(ContextInterface $context): int
    {
        try {
            $customer = $this->getCustomer->execute($context);

            return (int)$customer->getId();
        } catch (GraphQlNoSuchEntityException $exception) {
            return 0;
        }
    }
}
