<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package RMA GraphQl for Magento 2 (System)
 */

namespace Amasty\RmaGraphQl\Model\Request;

use Amasty\Rma\Api\CustomerRequestRepositoryInterface;
use Amasty\Rma\Api\Data\RequestInterface;
use Amasty\Rma\Controller\FrontendRma;
use Amasty\Rma\Model\ConfigProvider;
use Amasty\RmaGraphQl\Model\Resolver\Validator;
use Magento\Framework\App;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Sales\Api\OrderRepositoryInterface;

class CreateReturnRequest
{
    public const ORDER_ID_KEY = 'order_id';
    public const POLICY_KEY = 'policy_allowed';
    public const RETURN_ITEMS_KEY = 'return_items';
    public const CUSTOM_FIELDS_KEY = 'custom_fields';

    /**
     * @var FrontendRma
     */
    private $frontendRma;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var App\RequestFactory
     */
    private $requestFactory;

    /**
     * @var Validator\PlaceRequest
     */
    private $placeRequestValidator;

    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var CustomerRequestRepositoryInterface
     */
    private $requestRepository;

    public function __construct(
        FrontendRma $frontendRma,
        ConfigProvider $configProvider,
        App\RequestFactory $requestFactory,
        Validator\PlaceRequest $placeRequestValidator,
        OrderRepositoryInterface $orderRepository,
        CustomerRequestRepositoryInterface $requestRepository
    ) {
        $this->frontendRma = $frontendRma;
        $this->configProvider = $configProvider;
        $this->requestFactory = $requestFactory;
        $this->placeRequestValidator = $placeRequestValidator;
        $this->orderRepository = $orderRepository;
        $this->requestRepository = $requestRepository;
    }

    public function execute(array $args): RequestInterface
    {
        $this->placeRequestValidator->validateInput($args ?? []);
        $isPolicyAllowed = $args['input'][self::POLICY_KEY] ?? 0;

        if ($this->configProvider->isReturnPolicyEnabled() && (int)$isPolicyAllowed !== 1) {
            throw new GraphQlInputException(__('You didn\'t agree to Privacy policy.'));
        }

        $orderId = (int)$args['input'][self::ORDER_ID_KEY] ?? 0;
        $order = $this->orderRepository->get($orderId);

        return $this->frontendRma->processNewRequest(
            $this->requestRepository,
            $order,
            $this->prepareRequestData($args)
        );
    }

    private function prepareRequestData(array $args): App\RequestInterface
    {
        $httpRequest = $this->requestFactory->create();

        $items = [];
        foreach ($args['input'][self::RETURN_ITEMS_KEY] as $item) {
            $items[$item['order_item_id']] = $item;
        }
        $httpRequest->setParam('items', $items);

        if (isset($args['input'][self::CUSTOM_FIELDS_KEY])) {
            $customFields = [];
            foreach ($args['input'][self::CUSTOM_FIELDS_KEY] as $customFieldData) {
                $customFields[$customFieldData['key']] = $customFieldData['value'];
            }
            $httpRequest->setParam('custom_fields', $customFields);
        }

        return $httpRequest;
    }
}
