<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package RMA GraphQl for Magento 2 (System)
 */

namespace Amasty\RmaGraphQl\Model\File;

use Amasty\Rma\Model\ConfigProvider;
use Amasty\Rma\Utils\FileUpload;
use Amasty\RmaGraphQl\Api\Data\FileContentInterface;
use Amasty\RmaGraphQl\Api\UploadFilesInterface;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Filesystem;
use Magento\Framework\Math\Random;

class Upload implements UploadFilesInterface
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var Filesystem\Directory\WriteInterface
     */
    private $mediaWriter;

    /**
     * @var Random
     */
    private $mathRandom;

    public function __construct(
        ConfigProvider $configProvider,
        Filesystem $filesystem,
        Random $mathRandom
    ) {
        $this->configProvider = $configProvider;
        $this->mediaWriter = $filesystem->getDirectoryWrite(DirectoryList::MEDIA);
        $this->mathRandom = $mathRandom;
    }

    public function upload(array $filesContents): array
    {
        foreach ($filesContents as $fileContent) {
            $this->uploadFile($fileContent);
        }

        return $filesContents;
    }

    /**
     * @param FileContentInterface $fileContent
     * @return void
     * @throws LocalizedException
     */
    private function uploadFile(FileContentInterface $fileContent): void
    {
        if (!in_array($fileContent->getExtension(), FileUpload::ALLOWED_EXTENSIONS)) {
            throw new LocalizedException(__('Invalid file extension.'));
        }
        //phpcs:ignore
        if (!($content = base64_decode($fileContent->getBase64EncodedData()))) {
            throw new LocalizedException(__('Image file decoding error.'));
        }

        $tmpPath = $this->mediaWriter->getAbsolutePath(FileUpload::MEDIA_PATH . 'temp/');
        $this->mediaWriter->create($tmpPath);
        do {
            $fileHash = $this->mathRandom->getUniqueHash() . '.' . $fileContent->getExtension();
            $filePath = $tmpPath . $fileHash;
        } while ($this->mediaWriter->isExist($filePath));

        $this->mediaWriter->getDriver()->filePutContents($filePath, $content);
        try {
            $isFileValid = $this->validateUploadedFile($filePath);
        } catch (\Exception $e) {
            $isFileValid = false;
        } finally {
            if (!$isFileValid) {
                $this->mediaWriter->delete($filePath);
                throw new LocalizedException(__('Something went wrong during file uploading.'));
            }
            $fileContent->setBase64EncodedData('');
            $fileContent->setFilehash($fileHash);
            $fileContent->setOriginalFileName(
                str_replace('.' . $fileContent->getExtension(), '', $fileContent->getOriginalFileName())
            );
        }
    }

    /**
     * @param string $filePath
     * @return bool
     * @throws LocalizedException
     */
    private function validateUploadedFile(string $filePath): bool
    {
        if (!$this->mediaWriter->getDriver()->isExists($filePath)) {
            throw new LocalizedException(__('File does not exist.'));
        }

        $fileStat = $this->mediaWriter->getDriver()->stat($filePath);
        $maxFileSize = $this->configProvider->getMaxFileSize();
        if (!isset($fileStat['size']) || $fileStat['size'] == 0
            || ($maxFileSize > 0 && $fileStat['size'] > $maxFileSize * 1024)
        ) {
            throw new LocalizedException(__('Wrong file size.'));
        }

        return true;
    }
}
