<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Reward Points MFTF 3 for Magento 2 (System)
 */

namespace Amasty\RewardsMFTF3\Console\Command;

use Amasty\Rewards\Api\Data\RewardsInterface;
use Amasty\Rewards\Api\RewardsRepositoryInterface;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Console\Cli;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class GetExpirationDate extends Command
{
    /**
     * @var State
     */
    private $state;

    /**
     * @var RewardsRepositoryInterface
     */
    private $rewardsRepository;

    /**
     * @var RewardsInterface
     */
    private $rewards;

    public function __construct(
        State $state,
        RewardsRepositoryInterface $rewardsRepository,
        RewardsInterface $rewards
    ) {
        $this->state = $state;
        $this->rewardsRepository = $rewardsRepository;
        $this->rewards = $rewards;
        parent::__construct();
    }

    protected function configure()
    {
        parent::configure();
        $this->setName('mftf:rewards:get-expiration')
            ->setDescription('Get reward points expiration date.');

        $this->setDefinition(
            [
                new InputArgument(
                    'customer_id',
                    InputArgument::REQUIRED,
                    "Customer Id for change points expiration date"
                ),
                new InputArgument(
                    'inverse_transaction_number',
                    InputArgument::OPTIONAL,
                    "Inverse transaction number from current where rewards points for client were added",
                    '1'
                )
            ]
        );

    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->state->setAreaCode(Area::AREA_FRONTEND);

        $customerId = $input->getArgument('customer_id');
        $transactionNumber = abs((int)$input->getArgument('inverse_transaction_number')) - 1;

        $customerRewards = array_reverse($this->rewardsRepository->getByCustomerId($customerId));

        try{
            $points = $customerRewards[$transactionNumber];
        } catch (\Exception $e) {
            $output->writeln('<info>Error while get value! Assert points transaction. Error message: </info>');
            $output->writeln('<info>' . $e->getMessage() . '</info>');

            return Cli::RETURN_FAILURE;
        }

        $curExpDate = $points->getData($this->rewards::EXPIRATION_DATE);
        if ($curExpDate == null) {
            $output->writeln("<info>null</info>");
        } else {
            $date = date('Y-m-d', strtotime($curExpDate));
            $output->writeln("<info>$date</info>");
        }

        return Cli::RETURN_SUCCESS;
    }
}
