<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Reward Points MFTF 3 for Magento 2 (System)
 */

namespace Amasty\RewardsMFTF3\Console\Command;

use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Model\Logger;
use Magento\Customer\Model\ResourceModel\Visitor\CollectionFactory;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Console\Cli;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class CustomerVisitUpdate extends Command
{
    const TEST_EMAIL = 'RewardPointsTest@example.com';
    const TODAY_DAY = 'today';
    const LAST_MONTH = '-1 months';
    const LAST_DAY = '-1 day';
    const INVALID_MESS = 'Invalid customer. Allowed use only for customers where contains test email';


    /**
     * @var State
     */
    private $state;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var Logger
     */
    private $logger;

    /**
     * @var CollectionFactory
     */
    private $visitorCollection;

    public function __construct(
        State $state,
        CustomerRepositoryInterface $customerRepository,
        Logger $logger,
        CollectionFactory $visitorCollection,
    ) {
        $this->state = $state;
        $this->customerRepository = $customerRepository;
        $this->logger = $logger;
        $this->visitorCollection = $visitorCollection;
        parent::__construct();
    }

    protected function configure()
    {
        parent::configure();
        $this->setName('mftf:rewards:customer-login')
            ->setDescription('Change last customer login date (-1 month).');

        $this->setDefinition(
            [
                new InputArgument(
                    'customer_id',
                    InputArgument::REQUIRED,
                    "Customer Id for changes last login date"
                )
            ]
        );

    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->state->setAreaCode(Area::AREA_FRONTEND);
        $customerId = $input->getArgument('customer_id');

        $customer = $this->customerRepository->getById($customerId);

        if (strpos($customer->getEmail(),self::TEST_EMAIL) === false
            || strtotime($customer->getCreatedAt()) <= strtotime('today'))
        {
            $message = self::INVALID_MESS . ' - ' . self::TEST_EMAIL . '.';
            $output->writeln("<info>$message</info>");

            return Cli::RETURN_FAILURE;
        }

        $lastDateLogin = $this->logger->get($customerId)->getLastLoginAt();
        $newDateLogin = date('Y-m-d H:i:s', strtotime(self::LAST_MONTH . ' ' . $lastDateLogin));
        $newDateCreate = date('Y-m-d H:i:s', strtotime(
            self::LAST_MONTH . ' ' . self::LAST_DAY . ' ' . $lastDateLogin
        ));
        $logData = ['last_login_at' => $newDateLogin];

        $customer->setCreatedAt($newDateCreate);
        $this->logger->log($customerId, $logData);

        $collect = $this->visitorCollection->create();
        $collect->addFieldToFilter(
            'customer_id', ['eq' => (int)$customerId]
        );

        /** @var $row \Magento\Customer\Model\Visitor */
        foreach ($collect as $row) {
            $row->setCreatedAt($newDateCreate);
            $row->setLastVisitAt($newDateLogin);
        }

        $collect->save();
        $output->writeln('<info>Script is finished.</info>');

        return Cli::RETURN_SUCCESS;
    }
}
