<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Reward Points MFTF 3 for Magento 2 (System)
 */

namespace Amasty\RewardsMFTF3\Console\Command;

use Amasty\Rewards\Api\Data\RewardsInterface;
use Amasty\Rewards\Api\RewardsRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Console\Cli;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class ChangeExpirationDate extends Command
{
    const TEST_EMAIL = 'RewardPointsTest@example.com';
    const INVALID_MESS = 'Invalid customer. Allowed use only for customers where contains test email';

    /**
     * @var State
     */
    private $state;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var RewardsRepositoryInterface
     */
    private $rewardsRepository;

    /**
     * @var RewardsInterface
     */
    private $rewards;

    public function __construct(
        State $state,
        RewardsRepositoryInterface $rewardsRepository,
        RewardsInterface $rewards,
        CustomerRepositoryInterface $customerRepository
    ) {
        $this->state = $state;
        $this->rewardsRepository = $rewardsRepository;
        $this->rewards = $rewards;
        $this->customerRepository = $customerRepository;
        parent::__construct();
    }

    protected function configure()
    {
        parent::configure();
        $this->setName('mftf:rewards:change-expiration')
            ->setDescription('Change reward points expiration date.');

        $this->setDefinition(
            [
                new InputArgument(
                    'customer_id',
                    InputArgument::REQUIRED,
                    "Customer Id for change points expiration date"
                ),
                new InputArgument(
                    'change_days',
                    InputArgument::OPTIONAL,
                    "Count days to deduct from current expire date (use before '+' for plus days)",
                    '1'
                ),
                new InputArgument(
                    'inverse_transaction_number',
                    InputArgument::OPTIONAL,
                    "Inverse transaction number from current where rewards points for client were added",
                    '1'
                )
            ]
        );

    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->state->setAreaCode(Area::AREA_FRONTEND);

        $customerId = $input->getArgument('customer_id');
        $customer = $this->customerRepository->getById($customerId);

        if (strpos($customer->getEmail(),self::TEST_EMAIL) === false
            || strtotime($customer->getCreatedAt()) <= strtotime('today'))
        {
            $message = self::INVALID_MESS . ' - ' . self::TEST_EMAIL . '.';
            $output->writeln("<info>$message</info>");

            return Cli::RETURN_FAILURE;
        }

        $transactionNumber = abs((int)$input->getArgument('inverse_transaction_number')) - 1;
        $daysChange = $input->getArgument('change_days');

        if (strpos($daysChange, '+') !== null){
            $daysChange = '-' . $daysChange;
        }

        $customerRewards = array_reverse($this->rewardsRepository->getByCustomerId($customerId));
        $points = $customerRewards[$transactionNumber];

        $curExpDate = $points->getData($this->rewards::EXPIRATION_DATE);

        if ($curExpDate == null) {
            $curExpDate = date('Y-m-d H:i:s', strtotime('today +1 hour'));
        }

        $newExpDate = date('Y-m-d H:i:s', strtotime($daysChange . ' day ' . $curExpDate));
        $points->setExpirationDate($newExpDate);

        $expAmount = $points->getExpiringAmount();

        if ($expAmount == 0){
            $points->setExpiringAmount($points->getAmount());
        }

        $this->rewardsRepository->save($points);
        $output->writeln('<info>Script is finished.</info>');

        return Cli::RETURN_SUCCESS;
    }
}
