<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Ui\Component\Listing\Column;

use Magento\Framework\View\Element\UiComponent\ContextInterface;
use Magento\Framework\View\Element\UiComponentFactory;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Block\Adminhtml\Order\View\Info;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;
use Magento\Ui\Component\Listing\Columns\Column;

class Link extends Column
{
    /**
     * @var Info
     */
    private $info;

    /**
     * @var CollectionFactory
     */
    private $orderCollectionFactory;

    public function __construct(
        ContextInterface $context,
        UiComponentFactory $uiComponentFactory,
        CollectionFactory $orderCollectionFactory,
        Info $info,
        array $components = [],
        array $data = []
    ) {
        parent::__construct($context, $uiComponentFactory, $components, $data);
        $this->info = $info;
        $this->orderCollectionFactory = $orderCollectionFactory;
    }

    /**
     * Prepare Data Source
     *
     * @param array $dataSource
     *
     * @return array
     */
    public function prepareDataSource(array $dataSource): array
    {
        $orders = $this->getListOrdersHash($dataSource);

        if (!$orders) {
            return $dataSource;
        }

        foreach ($dataSource['data']['items'] as &$item) {
            if (isset($orders[$item[$this->getName()]])) {
                $item[$this->getName()] = $this->getLinkHtml($orders[$item[$this->getName()]]);
            }
        }

        return $dataSource;
    }

    /**
     * @param OrderInterface $order
     *
     * @return string
     */
    private function getLinkHtml(OrderInterface $order): string
    {
        $link = $this->info->getViewUrl($order->getEntityId());

        return '<a target="_blank" href="' . $link . '">' . $order->getIncrementId() . '</a>';
    }

    /**
     * @param array $dataSource
     * @return Order[]
     */
    private function getListOrdersHash(array $dataSource): array
    {
        $orderCollection = $this->orderCollectionFactory->create();
        $orderIncrementIds = [];
        $items = $dataSource['data']['items'] ?? [];
        
        foreach ($items as $item) {
            $orderIncrementId = $item[$this->getName()] ?? '';
            if ($orderIncrementId) {
                $orderIncrementIds[] = $orderIncrementId;
            }
        }

        if (!$orderIncrementIds) {
            return [];
        }

        $orderCollection->addAttributeToFilter('increment_id', ['in' => $orderIncrementIds]);

        $orders = [];

        /** @var Order $order */
        foreach ($orderCollection->getItems() as $order) {
            $orders[$order->getIncrementId()] = $order;
        }

        return $orders;
    }
}
