<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Setup\Patch\Data;

use Amasty\RecurringPayments\Api\Data\ProductRecurringAttributesInterface;
use Amasty\RecurringPayments\Api\Data\ProductRecurringGroupInterface;
use Amasty\RecurringPayments\Model\Config\Source\SubscriptionPlan;
use Amasty\RecurringPayments\Model\Product\Source\AvailableSubscription;
use Magento\Catalog\Api\Data\ProductAttributeInterface;
use Magento\Catalog\Model\Product\Attribute\Backend\Boolean as BackendBoolean;
use Magento\Eav\Model\Entity\Attribute\ScopedAttributeInterface;
use Magento\Eav\Model\Entity\Attribute\Source\Boolean;
use Magento\Eav\Setup\EavSetup;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;

class CreateProductAttributes implements DataPatchInterface
{
    /**
     * @var EavSetupFactory
     */
    private $eavSetupFactory;

    /**
     * @var ModuleDataSetupInterface
     */
    private $moduleDataSetup;

    public function __construct(EavSetupFactory $eavSetupFactory, ModuleDataSetupInterface $moduleDataSetup)
    {
        $this->eavSetupFactory = $eavSetupFactory;
        $this->moduleDataSetup = $moduleDataSetup;
    }

    /**
     * @return array
     */
    public static function getDependencies()
    {
        return [];
    }

    /**
     * @return array
     */
    public function getAliases()
    {
        return [];
    }

    /**
     * @return $this
     */
    public function apply()
    {
        /** @var EavSetup $eavSetup */
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        $eavSetup->addAttributeGroup(
            ProductAttributeInterface::ENTITY_TYPE_CODE,
            min($eavSetup->getAllAttributeSetIds('catalog_product')),
            ProductRecurringGroupInterface::SUBSCRIPTION_SETTINGS_GROUP,
            1000
        );

        $eavSetup->addAttribute(
            ProductAttributeInterface::ENTITY_TYPE_CODE,
            ProductRecurringAttributesInterface::RECURRING_ENABLE,
            [
                'label' => 'Available by Subscription',
                'type' => 'varchar',
                'default_value' => AvailableSubscription::NO,
                'source' => AvailableSubscription::class,
                'sort_order' => 10,
                'input' => 'select',
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'group' => ProductRecurringGroupInterface::SUBSCRIPTION_SETTINGS_GROUP,
                'visible' => true,
                'required' => false
            ]
        );

        $eavSetup->addAttribute(
            ProductAttributeInterface::ENTITY_TYPE_CODE,
            ProductRecurringAttributesInterface::SUBSCRIPTION_ONLY,
            [
                'label' => 'Subscription only',
                'backend' => BackendBoolean::class,
                'input' => 'select',
                'source' => Boolean::class,
                'default_value' => Boolean::VALUE_NO,
                'sort_order' => 20,
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'group' => ProductRecurringGroupInterface::SUBSCRIPTION_SETTINGS_GROUP,
                'visible' => true,
                'required' => false,
                'note' => "Select 'No' to make this product available as either regular purchase or subscription. "
                    . "Select 'Yes' to make this product subscription only."
            ]
        );

        $eavSetup->addAttribute(
            ProductAttributeInterface::ENTITY_TYPE_CODE,
            ProductRecurringAttributesInterface::PLANS,
            [
                'label' => 'Subscription Plans Available to Customers',
                'type' => 'text',
                'source' => SubscriptionPlan::class,
                'sort_order' => 30,
                'input' => 'multiselect',
                'global' => ScopedAttributeInterface::SCOPE_WEBSITE,
                'group' => ProductRecurringGroupInterface::SUBSCRIPTION_SETTINGS_GROUP,
                'backend' => \Magento\Eav\Model\Entity\Attribute\Backend\ArrayBackend::class,
                'visible' => true,
                'required' => false,
                'note' => 'Select one or multiple plans your customers would be able to choose from when subscribing' .
                    'to the product. If you need to add more plans or modify existing ones, please visit Sales >' .
                    ' Amasty Subscriptions > Subscription Plans.'
            ]
        );

        return $this;
    }
}
