<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Observer\Model\Quote;

use Amasty\RecurringPayments\Model\Payment;
use Amasty\RecurringPayments\Model\QuoteValidate;
use Amasty\RecurringPayments\Model\Subscription\Create\CreateSubscriptionsProcessorInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Quote\Item;

class SubmitSuccess implements ObserverInterface
{
    /**
     * @var Payment
     */
    private $payment;

    /**
     * @var CartRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var QuoteValidate
     */
    private $quoteValidate;

    /**
     * @var \Magento\Framework\Event\ManagerInterface
     */
    private $eventManager;

    /**
     * @var CreateSubscriptionsProcessorInterface
     */
    private $createSubscriptionsProcessor;

    public function __construct(
        Payment $payment,
        CartRepositoryInterface $quoteRepository,
        QuoteValidate $quoteValidate,
        \Magento\Framework\Event\ManagerInterface $eventManager,
        CreateSubscriptionsProcessorInterface $createSubscriptionsProcessor
    ) {
        $this->payment = $payment;
        $this->quoteRepository = $quoteRepository;
        $this->quoteValidate = $quoteValidate;
        $this->eventManager = $eventManager;
        $this->createSubscriptionsProcessor = $createSubscriptionsProcessor;
    }

    /**
     * @param Observer $observer
     */
    public function execute(Observer $observer)
    {
        /** @var \Magento\Sales\Api\Data\OrderInterface $order */
        $order = $observer->getData('order');
        /** @var \Magento\Sales\Api\Data\OrderPaymentInterface $payment */
        $payment = $order->getData('payment');

        if (!in_array($payment->getMethod(), $this->payment->getSupportedMethods())) {
            return;
        }

        $quoteItems = $this->quoteRepository->get($order->getQuoteId())->getAllVisibleItems();

        if (!$quoteItems) {
            return;
        }

        $subscriptionItems = [];
        /** @var Item $item */
        foreach ($quoteItems as $item) {
            if ($this->quoteValidate->validateQuoteItem($item)) {
                $subscriptionItems[] = $item;
            }
        }

        if ($subscriptionItems) {
            $args = [
                'order'              => $order,
                'subscription_items' => $subscriptionItems
            ];
            $this->eventManager->dispatch('amasty_recurring_order_placed_' . $payment->getMethod(), $args);
            $this->createSubscriptionsProcessor->process($order, $subscriptionItems);
        }
    }
}
