<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Subscription\Email;

use Amasty\RecurringPayments\Api\Subscription\NotificationInterface;
use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Model\Amount;
use Amasty\RecurringPayments\Model\Subscription\Scheduler\DateTimeInterval;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\View\Asset\Repository as AssetRepository;
use Magento\Framework\View\DesignInterface;

class RecurringTemplateVars
{
    public const TRIAL_IMAGE = 'trial_image';
    public const IS_INITIAL_FEE = 'is_initial_fee';

    /**
     * @var Amount
     */
    private $amount;

    /**
     * @var TimezoneInterface
     */
    private $timezone;

    /**
     * @var DateTimeInterval
     */
    private $dateTimeInterval;

    /**
     * @var AssetRepository
     */
    private $assetRepository;

    public function __construct(
        Amount $amount,
        TimezoneInterface $timezone,
        DateTimeInterval $dateTimeInterval,
        AssetRepository $assetRepository
    ) {
        $this->amount = $amount;
        $this->timezone = $timezone;
        $this->dateTimeInterval = $dateTimeInterval;
        $this->assetRepository = $assetRepository;
    }

    public function getRecurringData(SubscriptionInterface $subscription): array
    {
        $initialFee = (float)$subscription->getInitialFee();

        $isTrial = $this->isTrialPeriodActive($subscription);
        $trialKey = $isTrial ? 'yes' : 'no';

        return [
            self::TRIAL_IMAGE => $this->getTrialImage($trialKey),
            self::IS_INITIAL_FEE => $initialFee > 0,
            NotificationInterface::INTERVAL => $subscription->getDelivery(),
            NotificationInterface::INITIAL_FEE => $this->getOriginalPrice($initialFee),
            NotificationInterface::DISCOUNT_AMOUNT => $subscription->getBaseDiscountAmount(),
            NotificationInterface::DISCOUNT_CYCLE => $subscription->getRemainingDiscountCycles(),
            NotificationInterface::TRIAL_STATUS => $isTrial,
            NotificationInterface::TRIAL_DAYS => $subscription->getTrialDays(),
            NotificationInterface::START_DATE => $this->getStartDate($subscription),
            NotificationInterface::REGULAR_PAYMENT =>
                $this->getOriginalPrice((float)$subscription->getBaseGrandTotal()),
            NotificationInterface::PAYMENT_WITH_DISCOUNT =>
                $this->getOriginalPrice((float)$subscription->getBaseGrandTotalWithDiscount())
        ];
    }

    public function isTrialPeriodActive(SubscriptionInterface $subscription): bool
    {
        return $this->dateTimeInterval->isTrialPeriodActive(
            (string)$subscription->getStartDate(),
            (int)$subscription->getTrialDays()
        );
    }

    public function getStartDate(SubscriptionInterface $subscription): string
    {
        $timezoneName = $subscription->getCustomerTimezone();
        if (!$timezoneName) {
            $timezoneName = 'UTC';
        }
        $startDateObject = new \DateTime($subscription->getStartDate(), new \DateTimeZone('UTC'));
        $startDateObject->setTimezone(new \DateTimeZone($timezoneName));

        return $this->timezone->formatDate($startDateObject, \IntlDateFormatter::MEDIUM);
    }

    public function getOriginalPrice(float $price): string
    {
        return (string)$this->amount->convertAndFormat($price);
    }

    private function getTrialImage(string $trialKey): string
    {
        $params = [
            '_secure' => true,
            'area' => DesignInterface::DEFAULT_AREA
        ];

        return $this->assetRepository->getUrlWithParams(
            'Amasty_RecurringPayments/images/' . $trialKey . '-status.png',
            $params
        );
    }
}
