<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Subscription\Email;

use Magento\Bundle\Model\Product\Type;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\Data\ProductInterfaceFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Helper\Image;
use Magento\Catalog\Model\Product\Visibility;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\GroupedProduct\Model\Product\Type\Grouped;

class ProductTemplateVars
{
    public const FIRST_PARENT_PRODUCT = 0;
    public const BASE_IMAGE_ID = 'product_base_image';

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var ProductInterfaceFactory
     */
    private $productFactory;

    /**
     * @var Configurable
     */
    private $configurableType;

    /**
     * @var Type
     */
    private $bundleType;

    /**
     * @var Grouped
     */
    private $groupedType;

    /**
     * @var Image
     */
    private $imageHelper;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        ProductInterfaceFactory $productFactory,
        Configurable $configurableType,
        Type $bundleType,
        Grouped $groupedType,
        Image $imageHelper
    ) {
        $this->productRepository = $productRepository;
        $this->productFactory = $productFactory;
        $this->configurableType = $configurableType;
        $this->bundleType = $bundleType;
        $this->groupedType = $groupedType;
        $this->imageHelper = $imageHelper;
    }

    public function getProductData(int $productId)
    {
        try {
            /** @var ProductInterface $product */
            $product = $this->productRepository->getById($productId);
        } catch (NoSuchEntityException $exception) {
            $product = $this->productFactory->create();
        }

        return [
            'product_id' => (int)$product->getId(),
            'product_name' => $product->getName(),
            'product_image' => $this->getProductImage($product),
            'product_url' => $this->getProductUrl($product)
        ];
    }

    /**
     * @return string
     * @throws NoSuchEntityException
     */
    public function getProductUrl($product): string
    {
        if ($product->getVisibility() == Visibility::VISIBILITY_NOT_VISIBLE) {
            $parentProductIds = $this->getParentIdsByChild((int)$product->getId());

            if (!empty($parentProductIds[self::FIRST_PARENT_PRODUCT])) {
                /** @var ProductInterface $product */
                $product = $this->productRepository->getById($parentProductIds[self::FIRST_PARENT_PRODUCT]);
            }
        }

        return (string)$product->getUrlModel()->getUrl($product);
    }

    /**
     * @return string
     */
    public function getProductImage($product): string
    {
        $imgHelper = $this->imageHelper->init($product, self::BASE_IMAGE_ID);
        $imgHelper->setImageFile($product->getSmallImage());

        return (string)$imgHelper->getUrl();
    }

    /**
     * @param int $productId
     * @return array
     */
    private function getParentIdsByChild(int $productId): array
    {
        $parentProductIds = [];

        if ($this->configurableType->getParentIdsByChild($productId)) {
            $parentProductIds = $this->configurableType->getParentIdsByChild($productId);
        } elseif ($this->bundleType->getParentIdsByChild($productId)) {
            $parentProductIds = $this->bundleType->getParentIdsByChild($productId);
        } elseif ($this->groupedType->getParentIdsByChild($productId)) {
            $parentProductIds = $this->groupedType->getParentIdsByChild($productId);
        }

        return $parentProductIds;
    }
}
