<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Subscription\Email;

use Amasty\RecurringPayments\Api\Subscription\NotificationInterface;
use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Model\Config;
use Magento\Framework\DataObject as DataObject;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\NoSuchEntityException;

class EmailNotifier
{
    /**
     * @var Config
     */
    protected $config;

    /**
     * @var ManagerInterface
     */
    private $eventManager;

    /**
     * @var RecurringTemplateVars
     */
    private $recurringTemplateVars;

    /**
     * @var ProductTemplateVars
     */
    private $productTemplateVars;

    /**
     * @var CustomerTemplateVars
     */
    private $customerTemplateVars;

    public function __construct(
        ManagerInterface $eventManager,
        Config $config,
        RecurringTemplateVars $recurringTemplateVars,
        ProductTemplateVars $productTemplateVars,
        CustomerTemplateVars $customerTemplateVars
    ) {
        $this->eventManager = $eventManager;
        $this->config = $config;
        $this->recurringTemplateVars = $recurringTemplateVars;
        $this->productTemplateVars = $productTemplateVars;
        $this->customerTemplateVars = $customerTemplateVars;
    }

    /**
     * @param SubscriptionInterface $subscription
     * @param string $template
     * @param array $templateVariables
     */
    public function sendEmail(
        SubscriptionInterface $subscription,
        string $template,
        array $templateVariables = []
    ) {
        $templateVariables = array_merge($this->prepareTemplateVariables($subscription), $templateVariables);
        $storeId = (int)$subscription->getStoreId();

        $data = new DataObject(
            [
                'template'           => $template,
                'store_id'           => $storeId,
                'email_recipient'    => $subscription->getCustomerEmail(),
                'email_sender'       => $this->config->getEmailSender($storeId),
                'template_variables' => $templateVariables
            ]
        );

        $this->eventManager->dispatch('amasty_recurring_send_email', ['email_data' => $data]);
    }

    /**
     * @param SubscriptionInterface $subscription
     * @return array
     */
    private function prepareTemplateVariables(SubscriptionInterface $subscription): array
    {
        $templateVariables['recurring_data'] = $this->recurringTemplateVars->getRecurringData($subscription);
        $templateVariables['product'] = $this->productTemplateVars->getProductData((int)$subscription->getProductId());
        $templateVariables['customer']
            = $this->customerTemplateVars->getCustomerData((int)$subscription->getCustomerId());

        return $templateVariables;
    }
}
