<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Subscription\Create;

use Amasty\RecurringPayments\Api\Generators\RecurringTransactionGeneratorInterface;
use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Model\Config;
use Amasty\RecurringPayments\Model\Config\Source\Status;
use Amasty\RecurringPayments\Model\Subscription\Email\EmailNotifier;
use Amasty\RecurringPayments\Model\SubscriptionManagement;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\Sales\Api\Data\OrderInterface;

class DefaultProcessor implements CreateSubscriptionsProcessorInterface
{
    /**
     * @var CreateSubscriptionHandlerInterface
     */
    private $createSubscriptionHandler;

    /**
     * @var SubscriptionManagement
     */
    private $subscriptionManagement;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var EmailNotifier
     */
    private $emailNotifier;

    /**
     * @var RecurringTransactionGeneratorInterface
     */
    private $recurringTransactionGenerator;

    public function __construct(
        CreateSubscriptionHandlerInterface $createSubscriptionHandler,
        SubscriptionManagement $subscriptionManagement,
        Config $config,
        EmailNotifier $emailNotifier,
        RecurringTransactionGeneratorInterface $recurringTransactionGenerator
    ) {
        $this->createSubscriptionHandler = $createSubscriptionHandler;
        $this->subscriptionManagement = $subscriptionManagement;
        $this->config = $config;
        $this->emailNotifier = $emailNotifier;
        $this->recurringTransactionGenerator = $recurringTransactionGenerator;
    }

    /**
     * @param OrderInterface $order
     * @param AbstractItem[] $items
     * @return SubscriptionInterface[]
     */
    public function process(OrderInterface $order, array $items): array
    {
        $subscriptions = [];
        foreach ($items as $item) {
            $subscription = $this->subscriptionManagement->generateSubscription(
                $order,
                $item
            );
            $this->createSubscriptionHandler->handle($order, $item, $subscription);

            $subscription = $this->subscriptionManagement->saveSubscription(
                $subscription,
                $order
            );

            $transactionId = $order->getPayment() ? (string)$order->getPayment()->getLastTransId() : '';
            $orderIncrementId = $order->getIncrementId();
            $this->recurringTransactionGenerator->generate(
                (float)$order->getBaseGrandTotal(),
                $orderIncrementId,
                $order->getOrderCurrencyCode(),
                $transactionId,
                Status::SUCCESS,
                $subscription->getSubscriptionId(),
                null,
                $orderIncrementId
            );

            if ($this->config->isNotifySubscriptionPurchased((int)$subscription->getStoreId())) {
                $template = $this->config->getEmailTemplateSubscriptionPurchased((int)$subscription->getStoreId());
                $this->emailNotifier->sendEmail(
                    $subscription,
                    $template
                );
            }

            $subscriptions[] = $subscription;
        }

        return $subscriptions;
    }
}
