<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Repository;

use Amasty\RecurringPayments\Api\Subscription\SubscriptionInterface;
use Amasty\RecurringPayments\Api\Subscription\RepositoryInterface;
use Amasty\RecurringPayments\Model\SubscriptionFactory;
use Amasty\RecurringPayments\Model\ResourceModel\Subscription as SubscriptionResource;
use Amasty\RecurringPayments\Model\ResourceModel\Subscription\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Ui\Api\Data\BookmarkSearchResultsInterfaceFactory;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class SubscriptionRepository implements RepositoryInterface
{
    /**
     * @var BookmarkSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var SubscriptionFactory
     */
    private $subscriptionFactory;

    /**
     * @var SubscriptionResource
     */
    private $subscriptionResource;

    /**
     * Model data storage
     *
     * @var array
     */
    private $subscriptions;

    /**
     * @var CollectionFactory
     */
    private $subscriptionCollectionFactory;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    public function __construct(
        BookmarkSearchResultsInterfaceFactory $searchResultsFactory,
        SubscriptionFactory $subscriptionFactory,
        SubscriptionResource $subscriptionResource,
        CollectionFactory $subscriptionCollectionFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->searchResultsFactory = $searchResultsFactory;
        $this->subscriptionFactory = $subscriptionFactory;
        $this->subscriptionResource = $subscriptionResource;
        $this->subscriptionCollectionFactory = $subscriptionCollectionFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritdoc
     */
    public function save(SubscriptionInterface $subscription)
    {
        try {
            if ($subscription->getId()) {
                $subscription = $this->getById($subscription->getId())->addData($subscription->getData());
            }
            $this->subscriptionResource->save($subscription);
            unset($this->subscriptions[$subscription->getId()]);
        } catch (\Exception $e) {
            if ($subscription->getId()) {
                throw new CouldNotSaveException(
                    __(
                        'Unable to save subscription with ID %1. Error: %2',
                        [$subscription->getId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotSaveException(__('Unable to save new subscription. Error: %1', $e->getMessage()));
        }

        return $subscription;
    }

    /**
     * @inheritdoc
     */
    public function getById($id)
    {
        if (!isset($this->subscriptions[$id])) {
            /** @var \Amasty\RecurringPayments\Model\Subscription $subscription */
            $subscription = $this->subscriptionFactory->create();
            $this->subscriptionResource->load($subscription, $id);
            if (!$subscription->getId()) {
                throw new NoSuchEntityException(__('Subscription with specified ID "%1" not found.', $id));
            }
            $this->subscriptions[$id] = $subscription;
        }

        return $this->subscriptions[$id];
    }

    /**
     * @inheritdoc
     */
    public function getBySubscriptionId($subscriptionId)
    {
        /** @var \Amasty\RecurringPayments\Model\Subscription $subscription */
        $subscription = $this->subscriptionFactory->create();
        $this->subscriptionResource->load($subscription, $subscriptionId, SubscriptionInterface::SUBSCRIPTION_ID);
        if (!$subscription->getId()) {
            throw new NoSuchEntityException(__('Subscription with specified ID "%1" not found.', $subscriptionId));
        }

        return $subscription;
    }

    /**
     * @inheritdoc
     */
    public function delete(SubscriptionInterface $subscription)
    {
        try {
            $this->subscriptionResource->delete($subscription);
            unset($this->subscriptions[$subscription->getId()]);
        } catch (\Exception $e) {
            if ($subscription->getId()) {
                throw new CouldNotDeleteException(
                    __(
                        'Unable to remove subscription with ID %1. Error: %2',
                        [$subscription->getId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotDeleteException(__('Unable to remove subscription. Error: %1', $e->getMessage()));
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteById($id)
    {
        $subscriptionModel = $this->getById($id);
        $this->delete($subscriptionModel);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria)
    {
        /** @var \Amasty\RecurringPayments\Model\ResourceModel\Subscription\Collection $subscriptionCollection */
        $subscriptionCollection = $this->subscriptionCollectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $subscriptionCollection);

        $subscriptions = [];
        /** @var SubscriptionInterface $subscription */
        foreach ($subscriptionCollection->getItems() as $subscription) {
            $subscriptions[] = $this->getById($subscription->getId());
        }

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($subscriptions);

        return $searchResults;
    }
}
