<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Quote\Validator;

use Amasty\RecurringPayments\Model\Config;
use Amasty\RecurringPayments\Model\Product;
use Amasty\RecurringPayments\Model\Quote\ItemDataRetriever;
use Amasty\RecurringPayments\Model\Subscription\Mapper\StartEndDateMapper;
use Amasty\RecurringPayments\Model\Subscription\Scheduler\DateTimeInterval;
use Magento\Quote\Api\Data\CartItemInterface;

class StartEndDateValidator
{
    public const ERROR_START_DATE = 1;
    public const ERROR_END_DATE = 2;
    public const ERROR_COUNT_CYCLES = 3;

    /**
     * @var Config
     */
    private $configProvider;

    /**
     * @var StartEndDateMapper
     */
    private $startEndDateMapper;

    /**
     * @var DateTimeInterval
     */
    private $dateTimeInterval;

    /**
     * @var Product
     */
    private $recurringProduct;

    /**
     * @var ItemDataRetriever
     */
    private $itemDataRetriever;

    public function __construct(
        Config $configProvider,
        StartEndDateMapper $startEndDateMapper,
        DateTimeInterval $dateTimeInterval,
        Product $recurringProduct,
        ItemDataRetriever $itemDataRetriever
    ) {
        $this->configProvider = $configProvider;
        $this->startEndDateMapper = $startEndDateMapper;
        $this->dateTimeInterval = $dateTimeInterval;
        $this->recurringProduct = $recurringProduct;
        $this->itemDataRetriever = $itemDataRetriever;
    }

    /**
     * @param CartItemInterface $item
     */
    public function validate(CartItemInterface $item)
    {
        if (!$this->configProvider->isAllowSpecifyStartEndDate()) {
            return;
        }

        if (!$this->validateStartDate($item)) {
            $item->addErrorInfo(
                'amasty_recurring',
                self::ERROR_START_DATE,
                __('Subscription Start Date incorrect')
            );
            $item->getQuote()->addErrorInfo(
                'amasty_recurring',
                'amasty_recurring',
                self::ERROR_START_DATE,
                __('Some of the product options are invalid')
            );
        }

        if (!$this->validateEndDate($item)) {
            $item->addErrorInfo(
                'amasty_recurring',
                self::ERROR_END_DATE,
                __('Subscription End Date incorrect')
            );
            $item->getQuote()->addErrorInfo(
                'amasty_recurring',
                'amasty_recurring',
                self::ERROR_END_DATE,
                __('Some of the product options are invalid')
            );
        }

        if (!$this->validateCountCycles($item)) {
            $item->addErrorInfo(
                'amasty_recurring',
                self::ERROR_COUNT_CYCLES,
                __('End by a cycle should be greater than 1')
            );
            $item->getQuote()->addErrorInfo(
                'amasty_recurring',
                'amasty_recurring',
                self::ERROR_COUNT_CYCLES,
                __('Some of the product options are invalid')
            );
        }
    }

    /**
     * @param CartItemInterface $item
     * @return bool
     */
    private function validateStartDate(CartItemInterface $item): bool
    {
        $startDate = $this->startEndDateMapper->getSpecifiedStartDate($item);
        if (!$startDate) {
            return true;
        }

        $now = new \DateTime('now', $startDate->getTimezone());
        $now->setTime(0, 0, 0); // compare with start of day

        if ($startDate < $now) {
            return false;
        }

        return true;
    }

    /**
     * @param CartItemInterface $item
     * @return bool
     */
    private function validateEndDate(CartItemInterface $item): bool
    {
        $endDate = $this->startEndDateMapper->getSpecifiedEndDate($item);
        if (!$endDate) {
            return true;
        }

        $startDate = $this->startEndDateMapper->getSpecifiedStartDate($item);
        !$startDate && $startDate = new \DateTime('now', $endDate->getTimezone());

        $plan = $this->itemDataRetriever->getPlan($item);
        if (!$plan) {
            return false;
        }
        $trialDays = $plan->getEnableTrial() && $plan->getTrialDays()
            ? $plan->getTrialDays()
            : 0;
        $startDateAfterTrial = $this->dateTimeInterval->getStartDateAfterTrial(
            $startDate->format('Y-m-d'),
            $trialDays
        );

        $countIntervals = $this->dateTimeInterval->getCountIntervalsBetweenDates(
            $startDateAfterTrial,
            $endDate->format('Y-m-d'),
            $plan->getFrequency(),
            $plan->getFrequencyUnit()
        );

        return $countIntervals > 0;
    }

    /**
     * @param CartItemInterface $item
     * @return bool
     */
    private function validateCountCycles(CartItemInterface $item)
    {
        $requestOptions = $item->getBuyRequest()->getData();

        if (array_key_exists(Product::COUNT_CYCLES, $requestOptions)) {
            $value = $requestOptions[Product::COUNT_CYCLES];
            if ($value < 2) {
                return false;
            }
        }

        return true;
    }
}
