<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Model\Discount;

use Amasty\Base\Model\MagentoVersion;
use Amasty\RecurringPayments\Api\Data\SubscriptionPlanInterface;
use Amasty\RecurringPayments\Model\Amount;
use Amasty\RecurringPayments\Model\Generators\QuoteGenerator;
use Amasty\RecurringPayments\Model\Quote\ItemDataRetriever;
use Amasty\RecurringPayments\Model\QuoteValidate;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Quote\Model\Quote\Item\AbstractItem;

class ItemDiscount
{
    /**
     * @var Amount
     */
    private $amount;

    /**
     * @var QuoteValidate
     */
    private $quoteValidate;

    /**
     * @var ItemDataRetriever
     */
    private $itemDataRetriever;

    /**
     * @var PriceCurrencyInterface
     */
    private $priceCurrency;

    /**
     * @var MagentoVersion
     */
    private $magentoVersion;

    public function __construct(
        Amount $amount,
        QuoteValidate $quoteValidate,
        ItemDataRetriever $itemDataRetriever,
        PriceCurrencyInterface $priceCurrency,
        MagentoVersion $magentoVersion
    ) {
        $this->amount = $amount;
        $this->quoteValidate = $quoteValidate;
        $this->itemDataRetriever = $itemDataRetriever;
        $this->priceCurrency = $priceCurrency;
        $this->magentoVersion = $magentoVersion;
    }

    public function setDiscount(AbstractItem $item): AbstractItem
    {
        $baseDiscountAmount = null;
        $buyRequest = $item->getBuyRequest();
        if (isset($buyRequest['base_discount_amount'])) {
            $baseDiscountAmount = (float)$buyRequest['base_discount_amount'];
        } elseif ($this->quoteValidate->validateQuoteItem($item)) {
            $plan = $this->itemDataRetriever->getPlan($item, false);
            if (!$item->getQuote()->getData(QuoteGenerator::DISABLE_DISCOUNT_FLAG)
                && $plan->getEnableDiscount()
                && $plan->getDiscountAmount()
                && !$this->skipEstimationDiscount($item, $plan)
            ) {
                $baseDiscountAmount = $this->amount->getAmount(
                    $item->getProduct(),
                    (float)$plan->getDiscountAmount(),
                    $plan->getDiscountType(),
                    $item
                );
            }
        }

        if (!empty($baseDiscountAmount)) {
            $baseDiscountAmount = min($baseDiscountAmount, $item->getBaseRowTotal());
            $item->setBaseDiscountAmount($baseDiscountAmount);
            $item->setDiscountAmount($this->amount->convert((float)$baseDiscountAmount));
            $magentoVersion = $this->magentoVersion->get();
            // magento 2.4.0 for bundles have only child discount
            if (version_compare($magentoVersion, '2.4.0', '>=')
                && $item->getHasChildren()
                && $item->isChildrenCalculated()
            ) {
                $this->distributeDiscount($item);
            }
        }

        return $item;
    }

    /**
     * Distribute discount at parent item to children items
     *
     * @param AbstractItem $item
     * @return $this
     */
    private function distributeDiscount(AbstractItem $item): self
    {
        $parentBaseRowTotal = $item->getBaseRowTotal();
        $keys = [
            'discount_amount',
            'base_discount_amount',
            'original_discount_amount',
            'base_original_discount_amount',
        ];
        $roundingDelta = [];
        foreach ($keys as $key) {
            //Initialize the rounding delta to a tiny number to avoid floating point precision problem
            $roundingDelta[$key] = 0.0000001;
        }

        foreach ($item->getChildren() as $child) {
            $ratio = $parentBaseRowTotal != 0 ? $child->getBaseRowTotal() / $parentBaseRowTotal : 0;
            foreach ($keys as $key) {
                if (!$item->hasData($key)) {
                    continue;
                }
                $value = $item->getData($key) * $ratio;
                $roundedValue = $this->priceCurrency->round($value + $roundingDelta[$key]);
                $roundingDelta[$key] += $value - $roundedValue;
                $child->setData($key, $roundedValue);
            }
        }

        foreach ($keys as $key) {
            $item->unsetData($key);
        }

        return $this;
    }

    /**
     * Disable discount for quote estimation if the only discount cycle is used on checkout
     *
     * @param AbstractItem $item
     * @param SubscriptionPlanInterface $plan
     * @return bool
     */
    private function skipEstimationDiscount(AbstractItem $item, SubscriptionPlanInterface $plan): bool
    {
        if (!$item->getQuote()->getData(QuoteGenerator::GENERATED_FLAG)) {
            return false;
        }

        if ($plan->getEnableTrial() && $plan->getTrialDays()) {
            return false;
        }

        return $plan->getEnableDiscountLimit()
            && $plan->getNumberOfDiscountCycles() <= 1;
    }
}
