<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Subscriptions & Recurring Payments for Magento 2 (System)
 */

namespace Amasty\RecurringPayments\Controller\Adminhtml\Plan;

use Amasty\RecurringPayments\Api\Data\SubscriptionPlanInterface;
use Amasty\RecurringPayments\Api\SubscriptionPlanRepositoryInterface;
use Amasty\RecurringPayments\Model\SubscriptionPlan;
use Amasty\RecurringPayments\Model\SubscriptionPlanFactory;
use Magento\Backend\App\Action;
use Magento\Backend\Model\View\Result\RedirectFactory;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\Request\DataPersistorInterface;
use Psr\Log\LoggerInterface;

class Save extends Action implements HttpPostActionInterface
{
    public const ADMIN_RESOURCE = 'Amasty_RecurringPayments::recurring_payments_subscription_plans';

    /**
     * @var DataPersistorInterface
     */
    private $dataPersistor;

    /**
     * @var SubscriptionPlanRepositoryInterface
     */
    private $repository;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var SubscriptionPlanFactory
     */
    private $subscriptionPlanFactory;

    /**
     * @var RedirectFactory
     */
    private $redirectFactory;

    public function __construct(
        Action\Context $context,
        SubscriptionPlanRepositoryInterface $repository,
        SubscriptionPlanFactory $subscriptionPlanFactory,
        DataPersistorInterface $dataPersistor,
        LoggerInterface $logger,
        RedirectFactory $redirectFactory
    ) {
        parent::__construct($context);
        $this->dataPersistor = $dataPersistor;
        $this->repository = $repository;
        $this->logger = $logger;
        $this->subscriptionPlanFactory = $subscriptionPlanFactory;
        $this->redirectFactory = $redirectFactory;
    }

    /**
     * Save Action
     */
    public function execute()
    {
        $data = $this->getRequest()->getPostValue();
        if ($data) {
            /** @var SubscriptionPlan $model */
            $model = $this->subscriptionPlanFactory->create();
            $planId = $this->getRequest()->getParam(SubscriptionPlanInterface::PLAN_ID);

            try {
                if ($planId) {
                    $model = $this->repository->getById((int)$planId);
                } else {
                    $planId = null;
                }

                $model->addData($data);
                $model->setPlanId($planId);

                $this->repository->save($model);

                $this->messageManager->addSuccessMessage(__('The Subscription Plan has been saved.'));

                if ($this->getRequest()->getParam('back')) {
                    return $this->redirectToEdit($model->getId());
                }
            } catch (\Magento\Framework\Exception\LocalizedException $e) {
                $this->messageManager->addErrorMessage($e->getMessage());

                return $this->redirectToEdit($planId, $data);
            } catch (\Exception $e) {
                $this->messageManager->addErrorMessage(
                    __('The Subscription Plan has not been saved. Please review the error log for the details.')
                );
                $this->logger->critical($e);
                $this->redirectToEdit($planId, $data);

                return $this->redirectToEdit($planId, $data);
            }
        }

        return $this->redirectFactory->create()->setPath('*/*/');
    }

    /**
     * Redirect to Edit or New and save $data to session
     *
     * @param int|null $planId
     * @param array|null $data
     */
    private function redirectToEdit($planId = null, $data = null)
    {
        if ($data) {
            $this->dataPersistor->set('amasty_recurring_payments_subscription_plan', $data);
        }
        if ($planId) {
            return $this->redirectFactory->create()
                ->setPath('*/*/edit', [SubscriptionPlanInterface::PLAN_ID => $planId]);
        }
        return $this->redirectFactory->create()->setPath('*/*/new');
    }
}
