<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Product Option Templates for Magento 2
 */

namespace Amasty\Prot\Model\ResourceModel\Catalog\Product\Option;

use Amasty\Prot\Api\Data\TemplateOptionInterface;
use Magento\Catalog\Model\Product\Option\Value as ValueModel;
use Magento\Catalog\Model\Product\Option\ValueFactory as ValueModelFactory;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Zend_Db_Expr;
use Zend_Db_Statement_Exception;

class DuplicateMagentoOptions
{
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var ValueModel
     */
    private $valueModel;

    public function __construct(ResourceConnection $resourceConnection, ValueModelFactory $valueModelFactory)
    {
        $this->resourceConnection = $resourceConnection;
        $this->valueModel = $valueModelFactory->create();
    }

    /**
     * @param int $oldProductId
     * @param int $newProductId
     * @return void
     * @throws Zend_Db_Statement_Exception
     */
    public function execute(int $oldProductId, int $newProductId): void
    {
        $optionsCond = $this->cloneOptions($oldProductId, $newProductId);
        foreach ($optionsCond as $oldOptionId => $newOptionId) {
            $this->cloneTitles($oldOptionId, $newOptionId);
            $this->clonePrices($oldOptionId, $newOptionId);
            $this->valueModel->duplicate($oldOptionId, $newOptionId);
        }
    }

    /**
     * Return array like [oldOptionId => newOptionId]
     *
     * @param int $oldProductId
     * @param int $newProductId
     * @return array
     * @throws Zend_Db_Statement_Exception
     */
    private function cloneOptions(int $oldProductId, int $newProductId): array
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $this->getTable('catalog_product_option');

        $optionsCond = [];
        $optionsData = [];

        // read and prepare original product options
        $select = $connection->select()->from(
            ['cpo' => $table]
        )->joinLeft(
            ['apor' => $this->getTable(TemplateOptionInterface::RELATION_TABLE)],
            sprintf('cpo.option_id = apor.%s', TemplateOptionInterface::OPTION_ID),
            []
        )->where(
            'product_id = ?',
            $oldProductId
        )->where(
            $connection->prepareSqlCondition(TemplateOptionInterface::PARENT_OPTION_ID, ['null' => true])
        );

        $query = $connection->query($select);
        while ($row = $query->fetch()) {
            $optionsData[$row['option_id']] = $row;
            $optionsData[$row['option_id']]['product_id'] = $newProductId;
            unset($optionsData[$row['option_id']]['option_id']);
        }

        // insert options to duplicated product
        foreach ($optionsData as $oldOptionId => $data) {
            $connection->insert($table, $data);
            $optionsCond[(int) $oldOptionId] = (int) $connection->lastInsertId($table);
        }

        return $optionsCond;
    }

    private function cloneTitles(int $oldOptionId, int $newOptionId): void
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $this->getTable('catalog_product_option_title');

        $select = $connection->select()->from(
            $table,
            [new Zend_Db_Expr($newOptionId), 'store_id', 'title']
        )->where(
            'option_id = ?',
            $oldOptionId
        );

        $insertSelect = $connection->insertFromSelect(
            $select,
            $table,
            ['option_id', 'store_id', 'title'],
            AdapterInterface::INSERT_ON_DUPLICATE
        );

        $connection->query($insertSelect);
    }

    private function clonePrices(int $oldOptionId, int $newOptionId): void
    {
        $connection = $this->resourceConnection->getConnection();
        $table = $this->getTable('catalog_product_option_price');

        $select = $connection->select()->from(
            $table,
            [new Zend_Db_Expr($newOptionId), 'store_id', 'price', 'price_type']
        )->where(
            'option_id = ?',
            $oldOptionId
        );

        $insertSelect = $connection->insertFromSelect(
            $select,
            $table,
            ['option_id', 'store_id', 'price', 'price_type'],
            AdapterInterface::INSERT_ON_DUPLICATE
        );
        $connection->query($insertSelect);
    }

    private function getTable(string $name): string
    {
        return $this->resourceConnection->getTableName($name);
    }
}
