<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Product Labels GraphQL for Magento 2 (System)
 */

namespace Amasty\LabelGraphQl\Test\GraphQl;

use Amasty\Label\Api\LabelRepositoryInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class GetAmLabelProviderTest extends GraphQlAbstract
{
    private const MAIN_RESPONSE_KEY = 'amLabelProvider';
    private const LABEL_NAME = 'Amasty Test Label';

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var LabelRepositoryInterface
     */
    private $labelRepository;

    protected function setUp(): void
    {
        parent::setUp();

        $objectManager = Bootstrap::getObjectManager();
        $this->productRepository = $objectManager->get(ProductRepositoryInterface::class);
        $this->searchCriteriaBuilder = $objectManager->get(SearchCriteriaBuilder::class);
        $this->labelRepository = $objectManager->get(LabelRepositoryInterface::class);
    }

    /**
     * @group amasty_label
     *
     * @magentoApiDataFixture Amasty_LabelGraphQl::Test/GraphQl/_files/am_create_label.php
     */
    public function testAmLabelProviderProduct(): void
    {
        $simpleId = (int)$this->productRepository->get('simple_product')->getId();
        $simplePrice = number_format((int)$this->productRepository->get('simple_product')->getPrice(), 2, '.');
        $virtualId = (int)$this->productRepository->get('virtual_product')->getId();
        $virtualPrice = number_format((int)$this->productRepository->get('virtual_product')->getPrice(), 2, '.');

        $assertArray = [
            [
                'items' => [
                    [
                        'customer_group_ids' => "32000",
                        'is_visible' => true,
                        'image' => null,
                        'name' => "Amasty Test Label",
                        'position' => "middle-left",
                        'product_id' => $simpleId,
                        'size' => null,
                        'style' => "font-size: 12px;",
                        'redirect_url' => "am_label_red_url_product",
                        'alt_tag' => "am_alt_prod_tag",
                        'tooltip_settings' => [
                            'color' => null,
                            'status' => 1,
                            'text' => null,
                            'text_color' => null
                        ],
                        'txt' => "am_label_product_text \$$simplePrice",
                        'label_id' => $this->getLabelId(self::LABEL_NAME)
                    ]
                ]
            ],
            [
                'items' => [
                    [
                        'customer_group_ids' => "32000",
                        'is_visible' => true,
                        'image' => null,
                        'name' => "Amasty Test Label",
                        'position' => "middle-left",
                        'product_id' => $virtualId,
                        'size' => null,
                        'style' => "font-size: 12px;",
                        'redirect_url' => "am_label_red_url_product",
                        'alt_tag' => "am_alt_prod_tag",
                        'tooltip_settings' => [
                            'color' => null,
                            'status' => 1,
                            'text' => null,
                            'text_color' => null
                        ],
                        'txt' => "am_label_product_text \$$virtualPrice",
                        'label_id' => $this->getLabelId(self::LABEL_NAME)
                    ]
                ]
            ]
        ];

        $response = $this->graphQlQuery($this->getQueryProduct(), ['productIds' => [$simpleId, $virtualId]]);

        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_RESPONSE_KEY], $assertArray);
    }

    /**
     * @group amasty_label
     *
     * @magentoApiDataFixture Amasty_LabelGraphQl::Test/GraphQl/_files/am_create_label.php
     */
    public function testAmLabelProviderCategory(): void
    {
        $simpleId = (int)$this->productRepository->get('simple_product')->getId();
        $virtualId = (int)$this->productRepository->get('virtual_product')->getId();

        $assertArray = [
            [
                'items' => [
                    [
                        'customer_group_ids' => "32000",
                        'is_visible' => true,
                        'image' => null,
                        'name' => "Amasty Test Label",
                        'position' => "bottom-right",
                        'product_id' => $simpleId,
                        'size' => null,
                        'style' => "font-size: 13px;",
                        'redirect_url' => "am_label_red_url_category",
                        'alt_tag' => "am_alt_cat_tag",
                        'tooltip_settings' => [
                            'color' => null,
                            'status' => 1,
                            'text' => null,
                            'text_color' => null
                        ],
                        'txt' => "am_label_category_text simple_product",
                        'label_id' => $this->getLabelId(self::LABEL_NAME)
                    ]
                ]
            ],
            [
                'items' => [
                    [
                        'customer_group_ids' => "32000",
                        'is_visible' => true,
                        'image' => null,
                        'name' => "Amasty Test Label",
                        'position' => "bottom-right",
                        'product_id' => $virtualId,
                        'size' => null,
                        'style' => "font-size: 13px;",
                        'redirect_url' => "am_label_red_url_category",
                        'alt_tag' => "am_alt_cat_tag",
                        'tooltip_settings' => [
                            'color' => null,
                            'status' => 1,
                            'text' => null,
                            'text_color' => null
                        ],
                        'txt' => "am_label_category_text virtual_product",
                        'label_id' => $this->getLabelId(self::LABEL_NAME)
                    ]
                ]
            ]
        ];

        $response = $this->graphQlQuery($this->getQueryCategory(), ['productIds' => [$simpleId, $virtualId]]);

        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_RESPONSE_KEY], $assertArray);
    }

    private function getLabelId(string $name, int $status = 1): int
    {
        $id = 1;

        $searchCriteriaLabel = $this->searchCriteriaBuilder
            ->addFilter('name', $name)
            ->addFilter('status', $status)
            ->create();
        $labels = $this->labelRepository->getList($searchCriteriaLabel)->getItems();

        foreach ($labels as $label) {
            $id = (int)$label->getLabelId();
            break;
        }

        return $id;
    }

    private function getQueryProduct(): string
    {
        return <<<'QUERY'
query AmLabelProvider ($productIds:[Int]) {
  amLabelProvider(productIds:$productIds, mode:PRODUCT){
    items{
      customer_group_ids
      image
      is_visible
      label_id
      name
      position
      product_id
      size
      style
      redirect_url
      alt_tag
      tooltip_settings{
        color
        status
        text
        text_color
      }
      txt
    }
  }
}
QUERY;
    }

    private function getQueryCategory(): string
    {
        return <<<'QUERY'
query AmLabelProvider ($productIds:[Int]) {
  amLabelProvider(productIds:$productIds, mode:CATEGORY){
    items{
      customer_group_ids
      image
      is_visible
      label_id
      name
      position
      product_id
      size
      style
      redirect_url
      alt_tag
      tooltip_settings{
        color
        status
        text
        text_color
      }
      txt
    }
  }
}
QUERY;
    }
}
