<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Extended Order Grid for Magento 2
 */

namespace Amasty\Ogrid\Model;

use Amasty\Ogrid\Helper\Data;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ObjectManager;
use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemRepositoryInterface;
use Magento\InventoryApi\Api\SourceRepositoryInterface;
use Magento\InventoryCatalogApi\Model\IsSingleSourceModeInterface;
use Magento\Sales\Api\Data\OrderItemInterface;

class QuantityPerSource
{
    /**
     * @var IsSingleSourceModeInterface
     */
    private $isSingleSourceMode;

    /**
     * @var SourceRepositoryInterface
     */
    private $sourceRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var SourceItemRepositoryInterface
     */
    private $sourceItemRepository;

    public function __construct(
        IsSingleSourceModeInterface $isSingleSourceMode,
        SourceRepositoryInterface $sourceRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SourceItemRepositoryInterface $sourceItemRepository
    ) {
        $this->isSingleSourceMode = $isSingleSourceMode;
        $this->sourceRepository = $sourceRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sourceItemRepository = $sourceItemRepository;
    }

    public function modifyData(array $data): array
    {
        if (0 === $data['totalRecords'] || true === $this->isSingleSourceMode->execute()) {
            return $data;
        }

        $data['items'] = $this->getSourceItemsData($data['items']);

        return $data;
    }

    private function getSourceItemsData(array $dataItems): array
    {
        $itemsSkus = [];

        foreach ($dataItems as &$item) {
            $orderItems = &$item['amasty_ogrid_items_ordered'];

            foreach ($orderItems as &$orderItem) {
                $sku = $orderItem[Data::OGRID_PRODUCT_SKU_COLUMN] ?? $orderItem[OrderItemInterface::SKU];
                if (!isset($itemsSkus[$sku])) {
                    $itemsSkus[$sku] = [];
                }
                $orderItem['amasty_ogrid_product_quantity_per_source'] = [];
            }
        }

        unset($item);

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SourceItemInterface::SKU, array_keys($itemsSkus), 'in')
            ->create();

        $sourceItems = $this->sourceItemRepository->getList($searchCriteria)->getItems();
        $sourcesBySourceCode = $this->getSourcesBySourceItems($sourceItems);

        foreach ($sourceItems as $sourceItem) {
            $sku = $sourceItem->getSku();
            $source = $sourcesBySourceCode[$sourceItem->getSourceCode()];
            $qty = (float)$sourceItem->getQuantity();
            $itemsSkus[$sku][] = sprintf('%s (%s): %s', $source->getName(), $source->getSourceCode(), $qty);
        }

        foreach ($dataItems as &$item) {
            $orderItems = &$item['amasty_ogrid_items_ordered'];

            foreach ($orderItems as &$orderItem) {
                $sku = $orderItem[Data::OGRID_PRODUCT_SKU_COLUMN] ?? $orderItem[OrderItemInterface::SKU];

                if (isset($itemsSkus[$sku])) {
                    $orderItem['amasty_ogrid_product_quantity_per_source'] = implode(
                        "; \r\n",
                        $itemsSkus[$sku]
                    );
                }
            }
        }

        return $dataItems;
    }

    private function getSourcesBySourceItems(array $sourceItems): array
    {
        $newSourceCodes = $sourcesBySourceCodes = [];

        foreach ($sourceItems as $sourceItem) {
            $newSourceCodes[$sourceItem->getSourceCode()] = $sourceItem->getSourceCode();
        }

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SourceInterface::SOURCE_CODE, array_keys($newSourceCodes), 'in')
            ->create();
        $sources = $this->sourceRepository->getList($searchCriteria)->getItems();

        foreach ($sources as $source) {
            $sourcesBySourceCodes[$source->getSourceCode()] = $source;
        }

        return $sourcesBySourceCodes;
    }
}
