<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Thank You Page for Magento 2
*/

namespace Amasty\ThankYouPage\Block\Onepage\Success\Types;

use Amasty\ThankYouPage\Api\ConfigCreateAccountInterface;
use Amasty\ThankYouPage\Model\Config\Blocks;
use Amasty\ThankYouPage\Model\Template\Filter;
use Magento\Customer\Model\AccountManagement;
use Magento\Framework\View\Element\Template;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Registration;
use Magento\Cms\Block\Block;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Sales\Model\Order;
use Magento\Store\Model\ScopeInterface;

class CreateAccount extends CustomAbstract implements TypesInterface
{
    public const BLOCK_CONFIG_NAME = 'create_account';
    public const XML_PATH_CONFIG = 'customer/address/';

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var Registration
     */
    private $registration;

    /**
     * @var EncryptorInterface
     */
    private $encryptor;

    public function __construct(
        Template\Context $context,
        ConfigCreateAccountInterface $config,
        Block $cmsBlock,
        CheckoutSession $checkoutSession,
        Registration $registration,
        EncryptorInterface $encryptor,
        Filter $filter,
        Blocks $blockConfig,
        array $data = []
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->registration = $registration;
        $this->encryptor = $encryptor;
        parent::__construct($context, $config, $cmsBlock, $filter, $blockConfig, $data);
    }

    /**
     * @return null
     */
    public function getCacheLifetime()
    {
        return null;
    }

    /**
     * @return bool
     */
    public function isEnabled(): bool
    {
        return $this->config->isBlockEnabled()
            && $this->registration->isAllowed()
            && !$this->getOrder()->getCustomerId();
    }

    /**
     * @return bool
     */
    public function isEmailFieldVisible(): bool
    {
        return $this->config->isEmailFieldVisible();
    }

    /**
     * @return bool
     */
    public function isTermsAndConditionsVisible(): bool
    {
        return $this->config->isTermsAndConditionsVisible();
    }

    /**
     * @return bool
     */
    public function isPasswordAutoGenerated(): bool
    {
        return $this->config->isPasswordAutoGenerated();
    }

    /**
     * @return string|null
     */
    public function getTermsText(): ?string
    {
        return $this->config->getTermsText();
    }

    /**
     * @return string
     */
    protected function getGroupPrefix(): string
    {
        return 'block_' . self::BLOCK_CONFIG_NAME;
    }

    /**
     * @return Order
     */
    private function getOrder(): Order
    {
        return $this->checkoutSession->getLastRealOrder();
    }

    /**
     * Encrypt order id so it is not possible to create an account for arbitrary order
     *
     * @return string
     */
    public function getFormActionUrl(): string
    {
        return $this->getUrl(
            'thankyoupage/customer/create',
            ['_secure' => true, 'order_id' => urlencode($this->encryptor->encrypt($this->getOrder()->getId()))]
        );
    }

    /**
     * @return string
     */
    public function getOrderEmail(): string
    {
        return $this->getOrder()->getCustomerEmail();
    }

    /**
     * Get minimum password length
     *
     * @return int
     */
    public function getMinimumPasswordLength(): int
    {
        return (int)$this->_scopeConfig->getValue(AccountManagement::XML_PATH_MINIMUM_PASSWORD_LENGTH);
    }

    /**
     * Get number of password required character classes
     *
     * @return int
     */
    public function getRequiredCharacterClassesNumber(): int
    {
        return (int)$this->_scopeConfig->getValue(AccountManagement::XML_PATH_REQUIRED_CHARACTER_CLASSES_NUMBER);
    }

    /**
     * @return int
     */
    public function getWidth(): int
    {
        return $this->blockConfig->getWidthByBlockId(self::BLOCK_CONFIG_NAME);
    }

    /**
     * @return string|null
     */
    public function getShowDob(): ?string
    {
        return $this->_scopeConfig->getValue(
            self::XML_PATH_CONFIG . 'dob_show',
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @inheritdoc
     */
    public function getJsLayout(): string
    {
        $this->jsLayout['components'] = null;
        if ($this->getShowDob()) {
            $this->jsLayout['components']['amasty-thank-you-page-dob'] =
                [
                    'component' => 'Amasty_ThankYouPage/js/view/success/date-of-birth',
                    'isRequiredComponent' => $this->getShowDob() === 'req'
                ];
        } else {
            unset($this->jsLayout['components']['amasty-thank-you-page-dob']);
        }

        return parent::getJsLayout();
    }
}
