<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Special Promotions Rules Duration by Amasty
 */

namespace Amasty\RulesDuration\Model\SalesRule;

use Amasty\Rules\Model\ResourceModel\Rule;
use Amasty\Rules\Model\RuleProvider;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Framework\App\Request\DataPersistorInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\EntityManager\Operation\ExtensionInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\SalesRule\Api\Data\RuleInterface as SalesRuleInterface;

class SaveHandler implements ExtensionInterface
{
    /**
     * @var Rule
     */
    private $ruleResource;

    /**
     * @var MetadataPool
     */
    private $metadataPool;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var DataPersistorInterface
     */
    private $dataPersistor;

    /**
     * @var ProductMetadataInterface
     */
    private $productMetadata;

    /**
     * @var RuleProvider
     */
    private $ruleProvider;

    /**
     * @var TimezoneInterface
     */
    private $date;

    public function __construct(
        Rule $ruleResource,
        MetadataPool $metadataPool,
        RequestInterface $request,
        DataPersistorInterface $dataPersistor,
        RuleProvider $ruleProvider,
        TimezoneInterface $date,
        ProductMetadataInterface $productMetadata
    ) {
        $this->ruleResource = $ruleResource;
        $this->metadataPool = $metadataPool;
        $this->request = $request;
        $this->dataPersistor = $dataPersistor;
        $this->productMetadata = $productMetadata;
        $this->ruleProvider = $ruleProvider;
        $this->date = $date;
    }

    /**
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     *
     * @param \Magento\SalesRule\Model\Rule|\Magento\SalesRule\Model\Data\Rule $entity
     * @param array $arguments
     *
     * @return \Magento\SalesRule\Model\Rule|\Magento\SalesRule\Model\Data\Rule
     * @throws LocalizedException
     */
    public function execute($entity, $arguments = [])
    {
        if ($this->productMetadata->getEdition() === 'Enterprise') {
            return $entity;
        }

        //Get date from request because $entity has invalid data due to time in the date
        $fromDate = $this->request->getParam('amrule_from_date');
        $toDate = $this->request->getParam('amrule_to_date');

        //when create rule via api or duplicate rule, request doesn't have data
        if (($fromDate === null) && ($toDate === null)) {
            $fromDate = $entity->getData('from_date');
            $toDate = $entity->getData('to_date');
        }

        if ($toDate && (strtotime($fromDate) >= strtotime($toDate))) {
            $this->request->setPostValue('from_date', $fromDate);
            $this->request->setPostValue('to_date', $toDate);
            $this->dataPersistor->set('sale_rule', $this->request->getPostValue());
            throw new LocalizedException(__('End Date must follow Start Date.'));
        }

        $linkField = $this->metadataPool->getMetadata(SalesRuleInterface::class)->getLinkField();
        $ruleLinkId = (int)$entity->getDataByKey($linkField);

        /** @var \Amasty\Rules\Model\Rule $amRule */
        $amRule = $this->ruleProvider->getAmruleByRuleId($ruleLinkId);

        if (!$fromDate) {
            $fromDate = $this->date->date(null, null, false)->format('Y-m-d H:i:s');
        }

        if (!$amRule->getData('salesrule_id')) {
            $amRule->setData('salesrule_id', $ruleLinkId);
        }

        $amRule->setData('from_datetime', $fromDate);
        $amRule->setData('to_datetime', $toDate);

        $this->ruleResource->save($amRule);

        return $entity;
    }
}
