<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Special Promotions Conditions by Amasty
 */

namespace Amasty\RulesConditions\Model\Rule\Condition;

use Amasty\Rules\Model\DiscountRegistry;
use Amasty\RulesConditions\Model\OptionProvider\SalesRuleOptionProvider;
use Magento\Framework\Model\AbstractModel;
use Magento\Rule\Model\Condition\AbstractCondition;
use Magento\Rule\Model\Condition\Context;

class SalesRule extends AbstractCondition
{
    /**
     * @var string
     */
    protected $_inputType = 'select';

    /**
     * @var SalesRuleOptionProvider
     */
    private $salesRuleOptionProvider;

    public function __construct(
        Context $context,
        SalesRuleOptionProvider $salesRuleOptionProvider,
        DiscountRegistry $discountRegistry,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->salesRuleOptionProvider = $salesRuleOptionProvider;
        $this->discountRegistry = $discountRegistry;
    }

    /**
     * @return string
     */
    public function getValueElementType()
    {
        return 'select';
    }

    /**
     * @return array
     */
    public function getValueSelectOptions()
    {
        if (!$this->hasData('value_select_options')) {
            $this->setData('value_select_options', $this->salesRuleOptionProvider->toOptionArray());
        }

        return $this->getData('value_select_options');
    }

    /**
     * @return string
     */
    public function asHtml()
    {
        $value = $this->getValueElementHtml() ?? '';

        return $this->getTypeElementHtml()
            . __(sprintf(__('Applied Cart Price Rule') . ' %s: %s', $this->getOperatorElementHtml(), $value))
            . $this->getRemoveLinkHtml();
    }

    public function loadOperatorOptions(): self
    {
        $this->setOperatorOption(
            [
                '()'  => __('is'),
                '!()' => __('is not'),
            ]
        );

        return $this;
    }

    /**
     * @param AbstractModel $model
     * @return bool
     */
    public function validate(AbstractModel $model)
    {
        return $this->validateAttribute(explode(',', (string)$model->getAppliedRuleIds()))
            && ($this->isPromoRule() || $this->ruleHasDiscount());
    }

    private function isPromoRule(): bool
    {
        return in_array($this->getValue(), $this->discountRegistry->getAmpromoRuleIds(), true);
    }

    private function ruleHasDiscount(): bool
    {
        $rules = $this->discountRegistry->getDiscount();
        $value = (int)$this->getValue();

        foreach ($rules as $ruleId => $ruleItemsAmount) {
            if (($value === $ruleId) && (array_sum($ruleItemsAmount) > 0)) {
                return true;
            }
        }

        return false;
    }
}
