<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Shop By Hyva Compatibility
 */

declare(strict_types=1);

namespace Amasty\ShopbyHyvaCompatibility\ViewModel;

use Magento\Framework\View\Element\Block\ArgumentInterface;
use Amasty\Shopby\Helper\Data;
use Amasty\Shopby\Model\Layer\FilterList;
use Amasty\ShopbyBase\Model\Detection\MobileDetect;
use Magento\Catalog\Model\Layer\Resolver;
use Magento\Framework\Module\Manager;
use Magento\Framework\Registry;
use Magento\Catalog\Model\Product\ProductList\ToolbarMemorizer;
use Magento\Framework\View\Element\Template\Context;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Catalog\Block\Product\ProductList\Toolbar;
use Magento\Framework\View\Page\Config;

class ProductListAjax implements ArgumentInterface
{
    public const CACHE_TAG = 'client_';

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Catalog\Model\Layer
     */
    private $layer;

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var Manager
     */
    private $moduleManager;

    /**
     * @var ToolbarMemorizer
     */
    private $toolbarMemorizer;

    /**
     * @var Toolbar
     */
    private $toolbar;

    /**
     * @var MobileDetect
     */
    private $mobileDetect;

    /**
     * @var Config $config
     */
    protected $config;

    /**
     * @param Context $context
     * @param Resolver $layerResolver
     * @param Data $helper
     * @param Registry $registry
     * @param Manager $moduleManager
     * @param ToolbarMemorizer $toolbarMemorizer
     * @param MobileDetect $mobileDetect
     * @param ScopeConfigInterface $scopeConfig
     * @param Toolbar $toolbar
     * @param Config $config
     */
    public function __construct(
        Context $context,
        Resolver $layerResolver,
        Data $helper,
        Registry $registry,
        Manager $moduleManager,
        ToolbarMemorizer $toolbarMemorizer,
        MobileDetect $mobileDetect,
        ScopeConfigInterface $scopeConfig,
        Toolbar $toolbar,
        Config $config
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->layer = $layerResolver->get();
        $this->helper = $helper;
        $this->registry = $registry;
        $this->moduleManager = $moduleManager;
        $this->toolbarMemorizer = $toolbarMemorizer;
        $this->mobileDetect = $mobileDetect;
        $this->toolbar = $toolbar;
        $this->config = $config;
    }

    /**
     * Get is GTM enabled
     *
     * @return bool
     */
    public function isGoogleTagManager()
    {
        return $this->moduleManager->isEnabled('Magento_GoogleTagManager');
    }

    /**
     * Get is Ajax enabled
     *
     * @return bool
     */
    public function canShowBlock()
    {
        return $this->helper->isAjaxEnabled();
    }

    /**
     * Get is Setting Ajax enabled
     *
     * @return bool
     */
    public function isAjaxSettingEnabled()
    {
        return $this->helper->isAjaxSettingEnabled();
    }

    /**
     * Get is select by click enabled
     *
     * @return int
     */
    public function submitByClick(): int
    {
        return $this->helper->collectFilters();
    }

    /**
     * Get view mode
     *
     * @return string[]
     */
    public function getIdentities()
    {
        return [self::CACHE_TAG . '_' . $this->mobileDetect->isMobile() ? 'mobile' : 'desktop'];
    }

    /**
     * Get is scroll up enabled
     *
     * @return int
     */
    public function scrollUp(): int
    {
        return (int) $this->scopeConfig->getValue('amshopby/general/ajax_scroll_up');
    }

    /**
     * Retrieve active filters
     *
     * @return array
     */
    protected function getActiveFilters(): array
    {
        $filters = $this->layer->getState()->getFilters();
        if (!is_array($filters)) {
            $filters = [];
        }
        return $filters;
    }

    /**
     * Retrieve Clear Filters URL
     *
     * @return string
     */
    public function getClearUrl(): string
    {
        return $this->helper->getAjaxCleanUrl($this->getActiveFilters());
    }

    /**
     * Get current category ID
     *
     * @return int
     */
    public function getCurrentCategoryId(): int
    {
        return (int) $this->helper->getCurrentCategory()->getId();
    }

    /**
     * Get is only obe category can be selected
     *
     * @return int
     */
    public function isCategorySingleSelect(): int
    {
        $allFilters = $this->registry->registry(FilterList::ALL_FILTERS_KEY, []);
        foreach ($allFilters as $filter) {
            if ($filter instanceof \Amasty\Shopby\Model\Layer\Filter\Category) {
                return (int) !$filter->isMultiselect();
            }
        }

        return 0;
    }

    /**
     * Get config
     *
     * @param string $path
     * @return string
     */
    public function getConfig($path)
    {
        return $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get GTM acc ID
     *
     * @return string
     */
    public function getGtmAccountId()
    {
        return $this->getConfig(\Magento\GoogleTagManager\Helper\Data::XML_PATH_CONTAINER_ID);
    }

    /**
     * Get is memorizing enabled for toolbar
     *
     * @return int
     */
    public function isMemorizingAllowed(): int
    {
        return (int) $this->toolbarMemorizer->isMemorizingAllowed();
    }

    /**
     * Retrieve widget options in json format
     *
     * @param array $customOptions Optional parameter for passing custom selectors from template
     * @return string
     */
    public function getToolbarWidgetOptions(array $customOptions = []): string
    {
        return $this->toolbar->getWidgetOptionsJson($customOptions);
    }

    /**
     * Get grouped product value
     *
     * @param int $attributeId
     * @param string $groupCode
     * @return string $value
     */
    public function getGroupsByAttributeId(int $attributeId, string $groupCode): string
    {
        return '';
    }

    /**
     * Check is set one column layout
     *
     * @return bool
     */
    public function isOneColumn(): bool
    {
        return $this->config->getPageLayout() === '1column';
    }
}
