<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package ILN Compatibility with Live Search for Magento 2 (System)
 */

namespace Amasty\ShopByLiveSearch\Plugin\LiveSearchAdapter\Model\Aggregation\PriceBucketHandler;

use Amasty\Shopby\Model\Source\DisplayMode;
use Amasty\ShopbyBase\Api\Data\FilterSettingRepositoryInterface;
use Magento\Framework\Search\Response\Aggregation\ValueFactory;
use Magento\Framework\Search\Response\Bucket;
use Magento\Framework\Search\Response\BucketFactory;
use Magento\LiveSearchAdapter\Model\Aggregation\BucketHandlerInterface;

class AddTotalAggregation
{
    /**
     * @var FilterSettingRepositoryInterface
     */
    private $filterSettingRepository;

    /**
     * @var BucketFactory
     */
    private $bucketFactory;

    /**
     * @var ValueFactory
     */
    private $valueFactory;

    public function __construct(
        FilterSettingRepositoryInterface $filterSettingRepository,
        BucketFactory $bucketFactory,
        ValueFactory $valueFactory
    ) {
        $this->filterSettingRepository = $filterSettingRepository;
        $this->bucketFactory = $bucketFactory;
        $this->valueFactory = $valueFactory;
    }

    /**
     * Append new metric needed for slider/from-to-only price filters.
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     * @see \Magento\LiveSearchAdapter\Model\Aggregation\PriceBucketHandler::getBucket
     */
    public function afterGetBucket(BucketHandlerInterface $bucketHandler, ?Bucket $bucket): ?Bucket
    {
        if ($bucket !== null && $this->isFromToDisplayMode($bucket->getName())) {
            return $this->getBucketWithTotalMetric($bucket);
        }

        return $bucket;
    }

    /**
     * Create new bucket with new metric.
     * New metric include data about total count, min & max values.
     */
    private function getBucketWithTotalMetric(Bucket $bucket): Bucket
    {
        $totalCount = 0;
        $min = PHP_INT_MAX;
        $max = 0;

        $bucketValues = $bucket->getValues();
        foreach ($bucketValues as $value) {
            $metrics = $value->getMetrics();
            if (!isset($metrics['value']) || !isset($metrics['count'])) {
                continue;
            }

            [$from, $to] = explode('_', $metrics['value']);
            $min = min($min, $from);
            $max = max($max, $to);
            $totalCount += $metrics['count'];
        }
        $bucketValues[] = $this->valueFactory->create([
            'value' => 'data',
            'metrics' => [
                'value' => 'data',
                'count' => $totalCount,
                'min' => $min,
                'max' => $max
            ]
        ]);

        return $this->bucketFactory->create([
            'name' => $bucket->getName(),
            'values' => $bucketValues
        ]);
    }

    private function isFromToDisplayMode(string $bucketName): bool
    {
        $attributeCode = preg_replace('@_bucket$@', '', $bucketName);
        $filterSetting = $this->filterSettingRepository->getByAttributeCode($attributeCode);

        return in_array($filterSetting->getDisplayMode(), [
            DisplayMode::MODE_SLIDER,
            DisplayMode::MODE_FROM_TO_ONLY
        ]) || $filterSetting->getAddFromToWidget();
    }
}
