<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Shop By Brand Compatibility with Hyva for Magento 2
 */

namespace Amasty\ShopByBrandHyvaCompatibility\Plugin\ViewModel;

use Amasty\Mostviewed\Block\Widget\Related;
use Amasty\ShopbyBase\Block\Product\AttributeIcon;
use Amasty\ShopbyBrand\Model\ConfigProvider;
use Amasty\ShopbyBrand\Model\Source\Tooltip;
use Amasty\ShopbyBrand\ViewModel\OptionProcessor;
use Hyva\Theme\Service\CurrentTheme;
use Hyva\Theme\ViewModel\ProductListItem as ListItem;
use Magento\Catalog\Model\Product;
use Magento\Framework\View\Element\BlockFactory;

class ProductListItemPlugin
{
    /** @var Product */
    private $product;

    /** @var ConfigProvider  */
    private $configProvider;

    /** @var OptionProcessor  */
    private $optionProcessor;

    /** @var BlockFactory */
    private $blockFactory;

    /** @var CurrentTheme  */
    private $currentTheme;

    /**
     * @param ConfigProvider $configProvider
     * @param OptionProcessor $optionProcessor
     * @param BlockFactory $blockFactory
     * @param CurrentTheme $currentTheme
     */
    public function __construct(
        ConfigProvider $configProvider,
        OptionProcessor $optionProcessor,
        BlockFactory $blockFactory,
        CurrentTheme $currentTheme
    ) {
        $this->configProvider = $configProvider;
        $this->optionProcessor = $optionProcessor;
        $this->blockFactory = $blockFactory;
        $this->currentTheme = $currentTheme;
    }

    /**
     * Render brand icon
     *
     * @param ListItem $original
     * @param string $html
     * @param Product $product
     *
     * @return string
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterGetProductPriceHtml(
        ListItem $original,
        string $html,
        Product $product
    ): string {
        if (!$this->currentTheme->isHyva()) {
            return $html;
        }

        $this->setProduct($product);
        return $html . $this->getLogoHtml();
    }

    /**
     * Add Brand Label to Amasty Related Block
     *
     * @param Related $original
     * @param string $html
     * @param Product $product
     *
     * @return string
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterGetBrandLogoHtml(
        Related $original,
        $html,
        Product $product
    ): string {
        $this->setProduct($product);
        return $html . $this->getLogoHtml();
    }

    /**
     * Get Logo HTML
     *
     * @return string
     */
    private function getLogoHtml(): string
    {
        $logoHtml = '';
        if ($this->isShowOnListing()) {
            $this->optionProcessor->setPageType(Tooltip::LISTING_PAGE);

            /** @var AttributeIcon $block */
            $block = $this->blockFactory->createBlock(
                AttributeIcon::class,
                [
                    'data' => [
                        AttributeIcon::KEY_ATTRIBUTE_CODES => $this->getAttributeCodes(),
                        AttributeIcon::KEY_OPTION_PROCESSOR => $this->optionProcessor,
                        AttributeIcon::KEY_PRODUCT => $this->getProduct(),
                        AttributeIcon::PAGE_TYPE => 'listing'
                    ]
                ]
            );

            $logoHtml = $block->toHtml();
        }

        return $logoHtml;
    }

    /**
     * Retrieve attribute code
     *
     * @return array
     */
    private function getAttributeCodes(): array
    {
        if ($code = $this->configProvider->getBrandAttributeCode()) {
            return [$code];
        }

        return [];
    }

    /**
     * Is show on listing
     *
     * @return bool
     */
    private function isShowOnListing(): bool
    {
        return $this->configProvider->isShowOnListing();
    }

    /**
     * Get product
     *
     * @return Product
     */
    private function getProduct(): Product
    {
        return $this->product;
    }

    /**
     * Set product
     *
     * @param Product $product
     */
    private function setProduct(Product $product): void
    {
        $this->product = $product;
    }
}
