<?php
declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Shopby Base Hyva Compatibility (System)
 */

namespace Amasty\ShopByBaseHyvaCompatibility\ViewModel;

use Amasty\ShopbyBase\Model\OptionSetting;
use Amasty\ShopbyBrand\ViewModel\OptionProcessor;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Framework\View\Element\BlockFactory;
use Magento\Framework\View\Element\Template;
use Magento\Store\Model\ScopeInterface;
use Amasty\ShopbyBrand\Model\Brand\BrandDataInterface;

class Tooltip implements ArgumentInterface
{
    /** @var BlockFactory  */
    private $blockFactory;

    /** @var ScopeConfigInterface  */
    private $scopeConfig;

    /**
     * @param BlockFactory $blockFactory
     * @param ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        BlockFactory $blockFactory,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->blockFactory = $blockFactory;
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * Render tooltip content
     *
     * @param array $setting
     * @return string
     */
    public function renderTooltip(array $setting): string
    {
        $content = $this->replaceCustomVariables(
            $this->getConfig('general/tooltip_content'),
            $setting['_']
        );

        if (empty($content)) {
            return '';
        }

        $block = $this->blockFactory->createBlock(Template::class);
        if (!empty($setting[OptionSetting::SMALL_IMAGE_ALT])) {
            $block->setAltText($setting[OptionSetting::SMALL_IMAGE_ALT]);
        }

        if (!empty($setting[OptionProcessor::TITLE])) {
            $block->setTitleText($setting[OptionProcessor::TITLE]);
        }

        if (!empty($setting[OptionProcessor::IMAGE_URL])) {
            $block->setTooltipCustomIcon($setting[OptionProcessor::IMAGE_URL]);
        }

        return $block
            ->setIsBrandsPage(!empty($setting['is_brands_page']))
            ->setTooltipContent($content)
            ->setTemplate('Amasty_ShopByBaseHyvaCompatibility::tooltip.phtml')
            ->toHtml();
    }

    /**
     * Retrieve config value
     *
     * @param string $path
     * @return string
     */
    private function getConfig(string $path): string
    {
        return $this->scopeConfig->getValue(
            'amshopby_brand/'.$path,
            ScopeInterface::SCOPE_STORE
        ) ?: '';
    }

    /**
     * Replace custom variables
     *
     * @param string $template
     * @param array|\Amasty\ShopbyBrand\Model\Brand\BrandDataInterface $item
     * @return string
     */
    public function replaceCustomVariables($template, $item)
    {
        preg_match_all('@\{(.+?)\}@', $template, $matches);

        if (isset($matches[1]) && is_array($matches[1])) {
            foreach ($matches[1] as $match) {
                $value = '';
                switch ($match) {
                    case 'title':
                        if (isset($item['label'])) {
                            $value = '<h3>' . $item['label'] . '</h3>';
                        }
                        break;
                    case 'description':
                    case 'short_description':
                        if (isset($item[$match]) && $item[$match]) {
                            $value = '<p>' . $item[$match] . '</p>';
                        }
                        break;
                    case 'small_image':
                    case 'image':
                        $imgUrl = $match == 'small_image' ? $item['img'] : $item['image'];
                        if (!empty($imgUrl)) {
                            $value = sprintf(
                                '<img class="am-brand-%s" src="%s" alt="Brand Image"/>',
                                $match,
                                $imgUrl
                            );
                        }
                        break;
                }
                $template = str_replace('{' . $match . '}', $value, $template);
            }

        }

        return strip_tags($template, '<img><p><h3><b><strong>');
    }
}
