<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Seo Rich Data Yotpo (System)
*/

declare(strict_types=1);

namespace Amasty\SeoRichDataYotpo\Model\Review\GetReviews;

use Amasty\SeoRichData\Model\Review\FormatRating;
use Amasty\SeoRichData\Model\Review\GetBestRating;
use Amasty\SeoRichData\Model\Review\GetReviews\GenerateReviewRichData;
use Amasty\SeoRichData\Model\Review\GetReviews\ReviewProviderInterface;
use Amasty\SeoRichData\Model\Review\ReviewBuilder;
use Amasty\SeoRichDataYotpo\Model\Review\GetAggregateRating\GetYotpoRating;
use Amasty\Yotpo\Model\Yotpo\Client;

class GetYotpoReviews implements ReviewProviderInterface
{
    public const REVIEWS_PER_REQUEST = 30;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var GenerateReviewRichData
     */
    private $generateReviewRichData;

    /**
     * @var ReviewBuilder
     */
    private $reviewBuilder;

    /**
     * @var FormatRating
     */
    private $formatRating;

    /**
     * @var GetBestRating
     */
    private $getBestRating;

    public function __construct(
        Client $client,
        GenerateReviewRichData $generateReviewRichData,
        ReviewBuilder $reviewBuilder,
        FormatRating $formatRating,
        GetBestRating $getBestRating
    ) {
        $this->client = $client;
        $this->generateReviewRichData = $generateReviewRichData;
        $this->reviewBuilder = $reviewBuilder;
        $this->formatRating = $formatRating;
        $this->getBestRating = $getBestRating;
    }

    /**
     * @param int $productId
     * @param int $storeIdFilter
     * @param int $numberReviews
     * @param int $formatRating
     * @return array
     */
    public function execute(int $productId, int $storeIdFilter, int $numberReviews, int $formatRating): array
    {
        $reviewsRichData = [];
        foreach ($this->loadReviews($productId, $storeIdFilter, $numberReviews) as $review) {
            if (isset($review['score'])) {
                $bestRating = $this->getBestRating->execute($formatRating);
                $ratingValue = $this->formatRating->execute(
                    (float) $review['score'],
                    GetYotpoRating::BEST_YOTPO_RATING,
                    $bestRating
                );
            } else {
                $bestRating = 0;
                $ratingValue = 0;
            }

            $this->reviewBuilder->setNickname($review['user']['display_name'] ?? '');
            $this->reviewBuilder->setCreatedAt($review['created_at'] ?? '');
            $this->reviewBuilder->setTitle($review['title'] ?? '');
            $this->reviewBuilder->setDetail($review['content'] ?? '');
            $this->reviewBuilder->setRatingValue($ratingValue);
            $this->reviewBuilder->setBestRating($bestRating);

            $reviewsRichData[] = $this->generateReviewRichData->execute($this->reviewBuilder->create());
        }

        return $reviewsRichData;
    }

    /**
     * @param int $productId
     * @param int $storeIdFilter
     * @param int $numberReviews
     * @return array
     */
    private function loadReviews(int $productId, int $storeIdFilter, int $numberReviews): array
    {
        $lastPage = ceil($numberReviews / static::REVIEWS_PER_REQUEST);
        $page = 1;

        $reviews = [];
        while ($page <= $lastPage) {
            if ($numberReviews <= static::REVIEWS_PER_REQUEST) {
                $numberReviewsToLoad = $numberReviews;
            } else {
                $numberReviewsToLoad = static::REVIEWS_PER_REQUEST;
                $numberReviews -= static::REVIEWS_PER_REQUEST;
            }
            $newReviews = $this->loadReviewsByPage($productId, $storeIdFilter, $numberReviewsToLoad, $page);
            array_push($reviews, ...$newReviews);
            $page++;

            if (count($newReviews) < static::REVIEWS_PER_REQUEST) {
                break; // determine last page on yotpo server found
            }
        }

        return $reviews;
    }

    /**
     * @param int $productId
     * @param int $storeIdFilter
     * @param int $numberReviews
     * @param int $page
     * @return array
     */
    private function loadReviewsByPage(int $productId, int $storeIdFilter, int $numberReviews, int $page): array
    {
        return $this->client->getProductReviews($productId, $storeIdFilter, $page, $numberReviews);
    }
}
