<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Search Rate Limiter by Amasty
 */

namespace Amasty\SearchRateLimiter\Model;

use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\ObjectManagerInterface;
use Symfony\Component\Lock\LockFactory;
use Symfony\Component\RateLimiter\RateLimiterFactory;

class RateLimiterAdapter
{
    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var string
     */
    private $policy;

    /**
     * @var string
     */
    private $storageInstance;

    public function __construct(
        ObjectManagerInterface $objectManager,
        ConfigProvider $configProvider,
        string $policy,
        string $storageInstance
    ) {
        $this->objectManager = $objectManager;
        $this->configProvider = $configProvider;
        $this->policy = $policy;
        $this->storageInstance = $storageInstance;
    }

    public function getLimiterFactory(string $id): RateLimiterFactory
    {
        if (!class_exists(RateLimiterFactory::class) || !class_exists(LockFactory::class)) {
            throw new LocalizedException(__(
                'Symfony library \'symfony/rate-limiter\' or \'symfony/lock\' are not found. '
                . 'Please run \'composer require symfony/rate-limiter\' and \'composer require symfony/lock\''
                . ' commands to install them.'
            ));
        }

        return $this->objectManager->create(
            RateLimiterFactory::class,
            ['config' => [
                'id' => $id,
                'policy' => $this->policy,
                'limit' => $this->configProvider->getRequestLimitationNumber(),
                'interval' => $this->configProvider->getRequestLimitationPeriod() . ' minutes'
            ], 'storage' => [
                'instance' => $this->storageInstance
            ]]
        );
    }
}
