<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Search Rate Limiter by Amasty
 */

namespace Amasty\SearchRateLimiter\Model;

use Amasty\Base\Model\GetCustomerIp;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\LocalizedException;
use Psr\Log\LoggerInterface;
use Symfony\Component\RateLimiter\LimiterInterface;

class RateLimiter
{
    public const SEARCH_LIMITER = 'am_search_limiter';

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var GetCustomerIp
     */
    private $customerIp;

    /**
     * @var RateLimiterAdapter
     */
    private $rateLimiterAdapter;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        RequestInterface $request,
        GetCustomerIp $customerIp,
        RateLimiterAdapter $rateLimiterAdapter,
        LoggerInterface $logger,
        ConfigProvider $configProvider
    ) {
        $this->request = $request;
        $this->customerIp = $customerIp;
        $this->rateLimiterAdapter = $rateLimiterAdapter;
        $this->logger = $logger;
        $this->configProvider = $configProvider;
    }

    public function getLimiter(string $id): ?LimiterInterface
    {
        if (!$this->configProvider->isSearchRequestsLimitationEnabled()) {
            return null;
        }

        try {
            $limiterFactory = $this->rateLimiterAdapter->getLimiterFactory($id);
        } catch (LocalizedException $e) {
            $this->logger->critical($e);

            return null;
        }

        $controller = $this->request->getControllerName();
        $action = $this->request->getActionName();
        $ip = $this->customerIp->getCurrentIp();

        return $limiterFactory->create($ip . $controller . $action);
    }
}
