<?php
declare(strict_types=1);
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Reward Points Hyva Compatibility by Amasty
 */

namespace Amasty\RewardsHyvaCompatibility\ViewModel\Catalog;

use Amasty\Rewards\Api\CheckoutHighlightManagementInterface;
use Amasty\Rewards\Api\Data\HighlightInterface;
use Amasty\Rewards\Api\GuestHighlightManagementInterface;
use Amasty\Rewards\Block\Frontend\Catalog\HighlightProduct as AmastyHighlightProduct;
use Magento\Customer\Model\SessionFactory as CustomerSessionFactory;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * ViewModel allows pass needed configuration to template
 */
class Highlight implements ArgumentInterface
{
    /**
     * @var GuestHighlightManagementInterface
     */
    private GuestHighlightManagementInterface $guestHighlightManagement;

    /**
     * @var CustomerSessionFactory
     */
    private CustomerSessionFactory $sessionFactory;

    /**
     * @var RequestInterface
     */
    private RequestInterface $request;

    /**
     * @var StoreManagerInterface
     */
    private StoreManagerInterface $storeManager;

    /**
     * @var UrlInterface
     */
    private UrlInterface $urlBuilder;

    /**
     * @var CheckoutHighlightManagementInterface
     */
    private CheckoutHighlightManagementInterface $checkoutHighlightManagement;

    /**
     * @param CustomerSessionFactory $sessionFactory
     * @param GuestHighlightManagementInterface $guestHighlightManagement
     * @param RequestInterface $request
     * @param StoreManagerInterface $storeManager
     * @param UrlInterface $urlBuilder
     * @param CheckoutHighlightManagementInterface $checkoutHighlightManagement
     */
    public function __construct(
        CustomerSessionFactory $sessionFactory,
        GuestHighlightManagementInterface $guestHighlightManagement,
        RequestInterface $request,
        StoreManagerInterface $storeManager,
        UrlInterface $urlBuilder,
        CheckoutHighlightManagementInterface $checkoutHighlightManagement
    ) {
        $this->guestHighlightManagement = $guestHighlightManagement;
        $this->sessionFactory = $sessionFactory;
        $this->request = $request;
        $this->storeManager = $storeManager;
        $this->urlBuilder = $urlBuilder;
        $this->checkoutHighlightManagement = $checkoutHighlightManagement;
    }

    /**
     * Check if we can show highlight for guest
     *
     * @param int $page
     * @return bool
     */
    public function isGuestHighlightAvailable(
        int $page = GuestHighlightManagementInterface::PAGE_PRODUCT
    ): bool {
        return $this->guestHighlightManagement->isVisible($page);
    }

    /**
     * Get highlight for guest
     *
     * @param int $page
     * @return HighlightInterface|null
     */
    public function getGuestHighlightData(int $page = GuestHighlightManagementInterface::PAGE_PRODUCT)
    {
        return $this->guestHighlightManagement->getHighlight($page);
    }

    /**
     * Retrieve data for checkout
     *
     * @param string $page
     * @return HighlightInterface|null
     */
    public function getHighlightCartData(string $page = CheckoutHighlightManagementInterface::CART)
    {
        if (!$this->checkoutHighlightManagement->isVisible($page)) {
            return null;
        }

        return $this->checkoutHighlightManagement->fillData();
    }

    /**
     * Get registration link
     *
     * @return null|string
     */
    public function getRegistrationLink()
    {
        return $this->guestHighlightManagement->getRegistrationLink();
    }

    /**
     * Check if customer is logged in
     *
     * @return boolean
     */
    public function isCustomerLoggedIn(): bool
    {
        return $this->sessionFactory->create()->isLoggedIn();
    }

    /**
     * This method is copied from
     *
     * Amasty\Rewards\Block\Frontend\Catalog\HighlightProduct::getProductId()
     *
     * @return string|null
     */
    public function getProductId(): ?string
    {
        return $this->request->getParam('product_id', null)
                ?: $this->request->getParam('id', null);
    }

    /**
     * This method is copied from
     *
     * Amasty\Rewards\Block\Frontend\Catalog\HighlightProduct::getRefreshUrl()
     *
     * @param string $type (product|category)
     * @return string
     * @throws NoSuchEntityException
     */
    public function getLoggedInCustomerRefreshUrl(string $type = 'product'): string
    {
        $path = '/';
        if ($this->isCustomerLoggedIn()) {
            $path = implode('/', [
                AmastyHighlightProduct::API_METHOD,
                $this->storeManager->getStore()->getCode(),
                AmastyHighlightProduct::API_VERSION,
                'rewards/mine/highlight',
                $type
            ]);
        } elseif ($this->guestHighlightManagement->isVisible($this->getTypePage($type))) {
            $path = implode('/', [
                AmastyHighlightProduct::API_METHOD,
                $this->storeManager->getStore()->getCode(),
                AmastyHighlightProduct::API_VERSION,
                'rewards/mine/guest-highlight',
                $type
            ]);
        }

        return $this->urlBuilder->getUrl($path);
    }

    /**
     * This method to get the correct type number
     *
     * @param string $type
     * @return int
     */
    protected function getTypePage(string $type = 'product'): int
    {
        switch ($type) {
            case 'category':
                return GuestHighlightManagementInterface::PAGE_CATEGORY;
            case 'cart':
                return GuestHighlightManagementInterface::PAGE_CART;
            case 'checkout':
                return GuestHighlightManagementInterface::PAGE_CHECKOUT;
            default:
                return GuestHighlightManagementInterface::PAGE_PRODUCT;
        }
    }
}
