<?php
declare(strict_types=1);
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Reward Points Hyva Compatibility by Amasty
 */

namespace Amasty\RewardsHyvaCompatibility\Plugin\ViewModel;

use Amasty\Rewards\Api\GuestHighlightManagementInterface;
use Amasty\Rewards\Block\Frontend\Catalog\HighlightCategory;
use Hyva\Theme\ViewModel\ProductListItem;
use Magento\Catalog\Model\Product;
use Magento\Customer\Model\SessionFactory as CustomerSessionFactory;
use Magento\Framework\View\LayoutInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Plugin for Hyva ProductListItem
 *
 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
 */
class ProductListItemPlugin
{
    /**
     * @var LayoutInterface
     */
    private LayoutInterface $layout;
    /**
     * @var GuestHighlightManagementInterface
     */
    private GuestHighlightManagementInterface $guestHighlightManagement;
    /**
     * @var StoreManagerInterface
     */
    private StoreManagerInterface $storeManager;
    /**
     * @var CustomerSessionFactory
     */
    private CustomerSessionFactory $sessionFactory;

    /**
     * @param LayoutInterface $layout
     * @param GuestHighlightManagementInterface $guestHighlightManagement
     * @param StoreManagerInterface $storeManager
     * @param CustomerSessionFactory $sessionFactory
     */
    public function __construct(
        LayoutInterface $layout,
        GuestHighlightManagementInterface $guestHighlightManagement,
        StoreManagerInterface $storeManager,
        CustomerSessionFactory $sessionFactory
    ) {
        $this->layout = $layout;
        $this->guestHighlightManagement = $guestHighlightManagement;
        $this->storeManager = $storeManager;
        $this->sessionFactory = $sessionFactory;
    }

    /**
     * Check if customer is logged in
     *
     * @return boolean
     */
    public function isCustomerLoggedIn(): bool
    {
        return $this->sessionFactory->create()->isLoggedIn();
    }

    /**
     * Add reward points to product list item
     *
     * @param ProductListItem $subject
     * @param string $result
     * @param Product $product
     * @return null|string
     */
    public function afterGetProductPriceHtml(
        ProductListItem $subject,
        string $result,
        Product $product
    ): string {
        if ($this->isCustomerLoggedIn()) {
            /** @var HighlightCategory $block */
            $block = $this->layout->getBlock('amasty.rewards.hyva.highlight.category');
        } elseif ($this->guestHighlightManagement->isVisible(GuestHighlightManagementInterface::PAGE_CATEGORY)) {
            /** @var HighlightCategory $block */
            $block = $this->layout->getBlock('amasty.rewards.highlight.product-guest');
        } else {
            return $result;
        }
        $block->setProductId($product->getId())
            ->setProductSku($product->getSku());

        $result .= $block->toHtml();

        return $result;
    }
}
