<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Amasty\RequestQuote\Api\QuoteRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class UpdateQuoteRemarksTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'updateQuoteRemarks';
    public const CUSTOMER = 'customer_quote@example.com';
    public const CUSTOMER_PASS = 'password';

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->quoteRepository = $objectManager->get(QuoteRepositoryInterface::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
        $this->maskedQuoteIdToQuoteId = $objectManager->get(MaskedQuoteIdToQuoteIdInterface::class);
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/customer.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/assign_quote_customer_to_quote.php
     */
    public function testUpdateQuoteRemarks(): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active');
        $remarks = 'Test GraphQl Update Remarks';

        $query = $this->getQuery($maskedQuoteId, $remarks);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertTrue($response[self::MAIN_QUERY_KEY]);

        $quote = $this->quoteRepository->get($this->maskedQuoteIdToQuoteId->execute($maskedQuoteId));
        $quoteRemarks = $quote->getQuoteCustomerNote();

        $this->assertEquals($remarks, $quoteRemarks);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(string $quoteId, string $remarks): string
    {
        return <<<MUTATION
mutation {
  updateQuoteRemarks (
    quoteId:"$quoteId",
    remarks:"$remarks"
    )
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
