<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\GraphQl\Quote\GetQuoteItemIdByReservedQuoteIdAndSku;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class UpdateQuoteItemsInCartTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'updateQuoteItemsInCart';

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var GetQuoteItemIdByReservedQuoteIdAndSku
     */
    private $getQuoteItemIdByReservedQuoteIdAndSku;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->getQuoteItemIdByReservedQuoteIdAndSku = $objectManager->get(
            GetQuoteItemIdByReservedQuoteIdAndSku::class
        );
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/add_two_products_to_cart.php
     */
    public function testUpdateQuoteItemsInCart(): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active');
        $cartItemId = $this->getQuoteItemIdByReservedQuoteIdAndSku->execute('test_order_active', 'simple_10');

        $newQty = 5;
        $newPrice = 5;
        $note = 'special_note';
        $remarks = 'special_guest_note';

        $variables = [
            'cartId' => $maskedQuoteId,
            'cartItemId' => $cartItemId,
            'quantity' => $newQty,
            'price' => $newPrice,
            'note' => $note,
            'remarks' => $remarks
        ];

        $fields = [
            'cart' => [
                'amastyQuote' => [
                    'notes' => [
                        'customer_note' => $remarks
                    ],
                    'status' => 0
                ],
                'prices' => [
                    'subtotal_excluding_tax' => [
                        'value' => 85
                    ]
                ],
                'items' => [
                    [
                        'quantity' => 5,
                        'product' => [
                            'name' => 'Simple Product 10'
                        ],
                        'amasty_fields' => [
                            'notes' => [
                                'customer_note' => 'special_note'
                            ]
                        ]
                    ],
                    [
                        'quantity' => 3,
                        'product' => [
                            'name' => 'Simple Product 20'
                        ],
                        'amasty_fields' => [
                            'notes' => [
                                'customer_note' => ''
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $query = $this->getQuery();
        $response = $this->graphQlMutation($query, $variables);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(): string
    {
        return <<<'MUTATION'
mutation UpdateQuoteItems (
    $cartId: String!,
    $cartItemId: Int,
    $quantity: Float,
    $price: Float,
    $note: String,
    $remarks: String
  ){
updateQuoteItemsInCart(
    input: {
        cart_id:$cartId
        cart_items: [{
            cartItemUpdateInput: {
                cart_item_id:$cartItemId
                quantity:$quantity
            }
            price:$price
            note:$note
        }],
        remarks:$remarks
    }
    ){
        cart {
            amastyQuote {
                notes {
                    customer_note
                }
                status
            }
            prices {
                subtotal_excluding_tax {
                    value
                }
            }
            items {
                quantity
                product {
                    name
                }
                amasty_fields {
                    notes {
                        customer_note
                    }
                }
            }
        }
    }
}
MUTATION;
    }
}
