<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Request a Quote GraphQl for Magento 2 (System)
 */

namespace Amasty\RequestAQuoteGraphql\Test\GraphQl;

use Magento\Framework\Exception\AuthenticationException;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\GraphQl\Quote\GetQuoteItemIdByReservedQuoteIdAndSku;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class SubmitQuoteTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'submitQuote';
    public const CUSTOMER = 'customer_quote@example.com';
    public const CUSTOMER_PASS = 'password';

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var GetQuoteItemIdByReservedQuoteIdAndSku
     */
    private $getQuoteItemIdByReservedQuoteIdAndSku;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->getQuoteItemIdByReservedQuoteIdAndSku = $objectManager->get(
            GetQuoteItemIdByReservedQuoteIdAndSku::class
        );
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/add_two_products_to_cart.php
     *
     */
    public function testSubmitChangeQuote(): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active');
        $cartItemId = $this->getQuoteItemIdByReservedQuoteIdAndSku->execute('test_order_active', 'simple_10');

        $newQty = 5;
        $email = 'graph_test_customer' . rand(1, 100000) . '@mail.com';

        $variables = [
            'email' => $email,
            'firstname' => "first_graph",
            'lastname' => "last_graph",
            'cartId' => $maskedQuoteId,
            'cartItemId' => $cartItemId,
            'quantity' => $newQty
        ];

        $fields = [
            'cart' => [
                'amastyQuote' => [
                    'status' => 1
                ],
                'items' => [
                    [
                        'quantity' => 5,
                        'product' => [
                            'name' => 'Simple Product 10'
                        ]
                    ],
                    [
                        'quantity' => 3,
                        'product' => [
                            'name' => 'Simple Product 20'
                        ]
                    ]
                ]
            ]
        ];

        $query = $this->getQueryChange();
        $response = $this->graphQlMutation($query, $variables);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
        $this->assertIsString($response[self::MAIN_QUERY_KEY]['token']);
    }

    /**
     * @group amasty_request_a_quote
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/enable_module.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/set_allowed_groups.php
     *
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/customer.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/active_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/assign_quote_customer_to_quote.php
     * @magentoApiDataFixture Amasty_RequestAQuoteGraphql::Test/GraphQl/_files/active_quote/add_two_products_to_cart.php
     *
     */
    public function testSubmitCustomerQuote(): void
    {
        $maskedQuoteId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_order_active');

        $fields = [
            'cart' => [
                'amastyQuote' => [
                    'status' => 1
                ],
                'items' => [
                    [
                        'quantity' => 2,
                        'product' => [
                            'name' => 'Simple Product 10'
                        ]
                    ],
                    [
                        'quantity' => 3,
                        'product' => [
                            'name' => 'Simple Product 20'
                        ]
                    ]
                ]
            ]
        ];

        $query = $this->getQueryCustomer($maskedQuoteId);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
        $this->assertNull($response[self::MAIN_QUERY_KEY]['token']);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQueryChange(): string
    {
        return <<<'MUTATION'
mutation SubmitQuote (
    $email: String,
    $firstname: String,
    $lastname: String,
    $cartId: String!,
    $cartItemId: Int,
    $quantity: Float
    ){
        submitQuote(
            input: {
                customerInput: {
                    email:$email,
                    firstname:$firstname,
                    lastname:$lastname
                }
                updateQuoteItemsInput: {
                    cart_id:$cartId,
                    cart_items: [{
                        cartItemUpdateInput: {
                            cart_item_id:$cartItemId
                            quantity:$quantity
                        }
                    }]
                }
            }
        ) {
            cart {
                amastyQuote {
                    status
                }
                items {
                    quantity
                    product {
                        name
                    }
                }
            }
        token
    }
}
MUTATION;
    }

    /**
     * Returns GraphQl query string
     */
    private function getQueryCustomer($cartId): string
    {
        return <<<MUTATION
mutation {
        submitQuote (
            input: {
                updateQuoteItemsInput: {
                    cart_id:"$cartId",
                    cart_items:[]
                }
            }
        ) {
            cart {
                amastyQuote {
                    status
                }
                items {
                    quantity
                    product {
                        name
                    }
                }
            }
        token
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
